<?php
/**
 * Mail format creation class
 *
 * This file defines YggMailFormatterJP class.
 * It includes methods to make mail data.
 *
 * @package YggDore.Base.Locale.JP
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggMailFormatter class
 */
require_once( "YggDore/Base/YggMailFormatter.php" );
/**
 * Require YggMimeHeader class
 */
require_once( "YggDore/Base/Locale/YggMimeHeader.php" );
/**
 * Require YggStringIconv class
 */
require_once( "YggDore/Base/Locale/YggStringIconv.php" );



/**
 * Mail format creation class
 *
 * This class has methods to make mail data.
 * A mail encoded to ISO-2022-JP is created.
 *
 * <code>
 * // Single part mail
 * $ph = popen( "/usr/sbin/sendmail -t -ihoge@hoge.com", "w" );
 * $mail = new YggMailFormatterJP();
 * $mail->start( $ph, YggMailFormatterJP::SINGLEPART );
 * $mail->toList( 
 *     array(
 *         array( "mail" => "hoge@yggdore.com", "name" => "hogehoe" ),
 *         array( "mail" => "test@yggdore.com" )
 *     )
 * );
 * $mail->from( "from@yggdore.com", "YggDore Tester" );
 * $mail->subject( "Title" );
 * $mail->message( "test\ntes." );
 * $mail->end();
 *
 * // Multi part mail
 * $mail = new YggMailFormatterJP();
 * $mail->start( $ph, YggMailFormatterJP::MULTIPART );
 * $mail->toList( 
 *     array(
 *         array( "mail" => "hoge@yggdore.com", "name" => "hogehoge"
 *         array( "mail" => "test@yggdore.com" )
 *     )
 * );
 * $mail->from( "webmaster@yggdore.com", "YggDore Webmaster" );
 * $mail->subject( "Title" );
 * $mail->boundary();
 * $mail->message( "test\ntes." );
 * $mail->boundary();
 * $mail->attachData( "test.txt", "file data" );
 * $mail->end();
 * </code>
 *
 * @package YggDore.Base.Locale.JP
 */
class YggMailFormatterJP extends YggMailFormatter {
	/**
	 * Create mail address list
	 *
	 * This method creates mail address from $data.<br />
	 * Mail format is [name <maddr>] if you specify name,
	 * otherwise mail format is [maddr].<br />
	 * Maximum length of mail address is YggMailFormatter::MAIL_MAXLEN.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : name (option)<br />
	 * mail => string : mail address
	 * @return string
	 */
	public static function createAddrList( $data, $head = "" )
	{
		$mxlsz = self::MIMEHEADER_LINESIZE - strlen( YGG_EOL );
		
		if( count( $data ) <= 0 ){
			throw new UnexpectedValueException;
		}
		elseif( strlen( $head ) > $mxlsz ){
			throw new UnexpectedValueException;
		}

		$rts   = $head;
		$lsz   = strlen( $head );
		$spsp  = "";
		$keyl  = array_keys( $data );
		$keylc = count( $keyl );
		for( $i = 0; $i < $keylc; $i++ ){
			$rec =& $data[$keyl[$i]];

			$isnm = false;
			if( isset( $rec['name'] ) && $rec['name'] != "" ){
				$isnm = true;
			}

			if( $isnm ){
				if( !self::_checkEOL( $rec['name'] ) ){
					throw new UnexpectedValueException;
				}
			}

			if( !self::checkGlobalAddr( $rec['mail'] ) ){
				throw new UnexpectedValueException;
			}

			// One name creation
			if( $isnm ){
				$mblen = iconv_strlen( $rec['name'], "UTF-8" );
				if( $mblen <= 0 ){
					throw new UnexpectedValueException;
				}

				// Get first line
				// Get line data that excludes line head size and
				// space for deviding line
				$idx  = 0;
				$ldat = YggMimeHeader::convLine(
					$rec['name'],
					0,
					$mxlsz - $lsz - strlen( $spsp ),
					"UTF-8",
					"ISO-2022-JP"
				);
				if( $ldat['dlen'] > 0 ){
					// If data is normal, set it
					// If data is illegal,
					// changes next line and retry setting
					$w     = $spsp . $ldat['mime'];
					$rts  .= $w;
					$lsz  += strlen( $w );
					$idx   = $ldat['dlen'];
				}

				$spsp = " ";

				// Create data more than line two
				$scsz = $mxlsz - strlen( $spsp );
				while( $idx < $mblen ){ 
					$rts .= YGG_EOL;
					$lsz  = 0;

					$ldat = YggMimeHeader::convLine(
						$rec['name'], $idx, $scsz, "UTF-8", "ISO-2022-JP"
					);
					if( $ldat['dlen'] <= 0 ){
						throw new UnexpectedValueException;
					}

					$w    = $spsp . $ldat['mime'];
					$rts .= $w;
					$lsz += strlen( $w );

					$idx += $ldat['dlen'];
				}
			}

			$csp = "";
			if( $i < $keylc - 1 ){
				$csp = ",";
			}

			// One mail address creation
			$w    = "<" . $rec['mail'] . ">" . $csp;
			$wsz  = $lsz + strlen( $spsp . $w );
			if( $wsz > $mxlsz ){
				$rts .= YGG_EOL;
				$lsz  = 0;
				$spsp = " ";
			}

			$w     = $spsp . $w;
			$rts  .= $w;
			$lsz  += strlen( $w );

			$spsp  = " ";
		}

		return $rts;
	}


	/**
	 * Create subject header
	 *
	 * This method creates subject header from $subject.
	 *
	 * @param string $subject
	 * @return string
	 */
	public static function createSubjectHeader( $subject )
	{
		$rts = self::_checkEOL( $subject );
		if( !$rts ){
			throw new UnexpectedValueException;
		}

		return YggMimeHeader::conv(
			$subject,
			"Subject: ",
			YggString::EOL,
			self::MIMEHEADER_LINESIZE,
			"UTF-8",
			"ISO-2022-JP"
		);
	}


	/**
	 * Create mail message header
	 *
	 * This method returns mail message header.
	 *
	 * @return string
	 */
	public static function createMessageHeader()
	{
		return "Content-Transfer-Encoding: 7bit" . YGG_EOL .
		       "Content-Type: text/plain; charset=ISO-2022-JP";
	}


	/**
	 * Create mail message
	 *
	 * This method adjusts new line code of $msg and returns it.
	 *
	 * @param string $msg Taget of message
	 * @return string
	 */
	public static function createMessage( $msg )
	{
		$msg = YggStringIconv::conv( $msg, "UTF-8", "ISO-2022-JP//TRANSLIT//IGNORE" );
		if( $msg == "" ){
			throw new UnexpectedValueException;
		}

		return $msg;
	}


	/**
	 * Create attach data header
	 *
	 * Create attaching header that has attach file name $fname.
	 *
	 * @param string $fname file name
	 * @return string
	 */
	public static function createAttachFileHeader( $fname )
	{
		$rts = self::_checkEOL( $fname );
		if( !$rts ){
			throw new UnexpectedValueException;
		}
		elseif( strpos( $fname, "\"" ) !== false ){
			throw new UnexpectedValueException;
		}

		$slen = iconv_strlen( $fname, "UTF-8" );
		if( $slen <= 0 ){
			throw new UnexpectedValueException;
		}

		$efni = YggMimeHeader::convLine( $fname, 0, 76, "UTF-8", "ISO-2022-JP" );
		if( $efni['dlen'] < $slen ){
			throw new UnexpectedValueException;
		}

		return self::_createAttachFileHeader( $efni['mime'] );
	}
}
?>
