<?php
/**
 * Tool for japanese mobile
 *
 * This file defines YggMobileJP class.
 * It includes methods to use for japanese mobile.
 *
 * @package YggDore.Base.Locale.JPM
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );
/**
 * Require YggMailFormatter class
 */
require_once( "YggDore/Base/YggMailFormatter.php" );




/**
 * Tool class for japanese mobile
 *
 * This class has methods to use for japanese mobile.
 *
 * @package YggDore.Base.Locale.JPM
 */
class YggMobileJP {
	/**
	 * Client type : Other
	 */
	const CT_OTHER = 0;
	/**
	 * Client type : Docomo(Japanese mobile)
	 */
	const CT_DOCOMO = 1;
	/**
	 * Client type : AU(Japanese mobile)
	 */
	const CT_AU = 2;
	/**
	 * Client type : Softbank(Japanese mobile)
	 */
	const CT_SOFTBANK = 3;


	/**
	 * Check e-mail address is for mobile
	 *
	 * @param string $mail E-mail
	 * @return boolean Return TRUE if $mail is mobile e-mail,
	 * otherwise return FALSE.
	 */
	public static function matchMail( $mail )
	{
		$rts = YggMailFormatter::checkGlobalAddr( $mail );
		if( !$rts ){
			return false;
		}

		$cptn = array(
			'/@docomo\.ne\.jp$/i',
			'/@ezweb\.ne\.jp$/i',
			'/@softbank\.ne\.jp$/i',
			'/@[a-z]\.vodafone\.ne\.jp$/i'
		);
		foreach( $cptn as $cpr ){
			if( preg_match( $cpr, $mail ) ){
				return true;
			}
		}

		return false;
	}


	/**
	 * Check client IP address matches japanese docomo mobile IP address
	 *
	 * @param string $cip Client IP address
	 * @return boolean Return TRUE if $cip is mobile IP address,
	 * otherwise return FALSE.
	 */
	public static function matchIPDocomo( $cip )
	{
		$cidrl = array(
			"210.153.84.0/24",
			"210.136.161.0/24",
			"210.153.86.0/24",
			"124.146.174.0/24",
			"124.146.175.0/24",
			"202.229.176.0/24",
			"202.229.177.0/24",
			"202.229.178.0/24"
		);
		
		return YggTool::matchNetAddr( $cip, $cidrl );
	}


	/**
	 * Check client IP address matches japanese au mobile IP address
	 *
	 * @param string $cip Client IP address
	 * @return boolean Return TRUE if $cip is mobile IP address,
	 * otherwise return FALSE.
	 */
	public static function matchIPAu( $cip )
	{
		$cidrl = array(
			"210.230.128.224/28",
			"121.111.227.160/27",
			"61.117.1.0/28",
			"219.108.158.0/27",
			"219.125.146.0/28",
			"61.117.2.32/29",
			"61.117.2.40/29",
			"219.108.158.40/29",
			"219.125.148.0/25",
			"222.5.63.0/25",
			"222.5.63.128/25",
			"222.5.62.128/25",
			"59.135.38.128/25",
			"219.108.157.0/25",
			"219.125.145.0/25",
			"121.111.231.0/25",
			"121.111.227.0/25",
			"118.152.214.192/26",
			"118.159.131.0/25",
			"118.159.133.0/25",
			"118.159.132.160/27"
		);
		
		return YggTool::matchNetAddr( $cip, $cidrl );
	}


	/**
	 * Check client IP address matches japanese softbank mobile IP address
	 *
	 * @param string $cip Client IP address
	 * @return boolean Return TRUE if $cip is mobile IP address,
	 * otherwise return FALSE.
	 */
	public static function matchIPSoftbank( $cip )
	{
		$cidrl = array(
			"123.108.236.0/24",
			"123.108.237.0/27",
			"202.179.204.0/24",
			"202.253.96.224/27",
			"210.146.7.192/26",
			"210.146.60.192/26",
			"210.151.9.128/26",
			"210.169.130.112/28",
			"210.175.1.128/25",
			"210.228.189.0/24",
			"211.8.159.128/25"
		);
		
		return YggTool::matchNetAddr( $cip, $cidrl );
	}


	/**
	 * Check client IP address matches japanese mobile IP address
	 *
	 * @param string $cip Client IP address
	 * @return boolean Return TRUE if $cip is mobile IP address,
	 * otherwise return FALSE.
	 */
	public static function matchIP( $cip )
	{
		return self::matchIPDocomo( $cip ) ||
		       self::matchIPAu( $cip ) ||
			   self::matchIPSoftbank( $cip );
	}


	/**
	 * Get client type from IP address
	 *
	 * Client type is checked from $ip.
	 * 
	 * @param string $ip IP address of client
	 * @return integer Client type<br />
	 * (It is one of the following options)<br />
	 * YggMobileJP::CT_DOCOMO : Docomo(Japanese mobile)<br />
	 * YggMobileJP::CT_AU : AU(Japanese mobile)<br />
	 * YggMobileJP::CT_SOFTBANK : Softbank(Japanese mobile)<br />
	 * YggMobileJP::CT_OTHER : Other
	 */
	public static function checkIPType( $cip )
	{
		if( self::matchIPDocomo( $cip ) ){
			return self::CT_DOCOMO;
		}
		elseif( self::matchIPAu( $cip ) ){
			return self::CT_AU;
		}
		elseif( self::matchIPSoftbank( $cip ) ){
			return self::CT_SOFTBANK;
		}

		return self::CT_OTHER;
	}
}
?>
