<?php
/**
 * String class
 *
 * This file defines YggStringJP class.
 * It includes methods to convert zenkaku and hankaku character and so on.
 *
 * @package YggDore.Base.Locale.JP
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );
/**
 * Require YggStringUTF8 class
 */
require_once( "YggDore/Base/Locale/YggStringUTF8.php" );



/**
 * String class
 *
 * This class has methods to convert zenkaku and hankaku character and so on.
 *
 * @package YggDore.Base.Locale.JP
 */
class YggStringJP extends YggStringUTF8 {
	/**
	 * Trim space
	 * 
	 * This method trims left or/and right space code.<br />
	 * This method supports to trim zenkaku space.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $data = "  Hello1!Good1!  ";
	 * $data = YggStringJP::trim( $data, YggStringJP::TRIM );
	 * (Result)
	 * $data => "Hello1!Good1!"
	 * </code>
	 *
	 * @param string $data Target string
	 * @param integer $adjf Trim mode<br />
	 * YggString::TRIM : Both trimming mode<br />
	 * YggString::LTRIM : Left trimming mode<br />
	 * YggString::RTRIM : Right trimming mode
	 * @return string
	 */
	static public function trim( $data, $adjf = self::TRIM )
	{
		if( !($adjf & 0x03) ){
			throw new UnexpectedValueException;
		}

		if( $data == "" ){
			return "";
		}

		if( $adjf & self::RTRIM ){
			$data = preg_replace( '/[[:space:]　]+$/u', "", $data );
		}
		if( $adjf & self::LTRIM ){
			$data = preg_replace( '/^[[:space:]　]+/u', "", $data );
		} 

		return (string)$data;
	}


	/**
	 * Replace multi spaces
	 * 
	 * This method replaces from multi spaces code to $sp in $data.<br />
	 * This method supports to replace zenkaku space.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $data = "  a   bc  10 ";
	 * $data = YggStringJP::spaceOne( $data );
	 * (Result)
	 * $data => " a bc 10 "
	 * </code>
	 *
	 * @param string $data Target data
	 * @param string $sp String of replacing to
	 * @return string
	 */
	public static function spaceOne( $data, $sp = " " )
	{
		if( $data == "" ){
			return "";
		}

		return (string)preg_replace( '/[[:space:]　]+/u', $sp, $data );
	}


	/**
	 * Convert ascii code
	 * 
	 * This method converts from ascii code to zenkaku or hankaku.
	 * If $isz2h is specified TRUE, characters in $data are converted to hankaku,
	 * otherwize to zenkaku.
	 *
	 * <code>
	 * $rts = YggStringJP::convASCII( "９８ａ" );
	 * // rts = "98a"
	 * </code>
	 *
	 * @param string $data Target data
	 * @param boolean $isz2h Zenkaku to hankaku mode
	 * @return string
	 */
	public function convASCII( $data, $isz2h = true )
	{
		$ctbl = array(
			array(
				"c1" => array( "ａ", "ｚ" ),
				"c2" => array( "a", "z" )
			),
			array(
				"c1" => array( "Ａ", "Ｚ" ),
				"c2" => array( "A", "Z" )
			),
			array(
				"c1" => array( "０", "９" ),
				"c2" => array( "0", "9" )
			),
			array(
				"c1" => array( "！", "！" ),
				"c2" => array( "!", "!" )
			),
			array(
				"c1" => array( "”", "”" ),
				"c2" => array( "\"", "\"" )
			),
			array(
				"c1" => array( "＃", "＃" ),
				"c2" => array( "#", "#" )
			),
			array(
				"c1" => array( "＄", "＄" ),
				"c2" => array( "$", "$" )
			),
			array(
				"c1" => array( "％", "％" ),
				"c2" => array( "%", "%" )
			),
			array(
				"c1" => array( "＆", "＆" ),
				"c2" => array( "&", "&" )
			),
			array(
				"c1" => array( "’", "’" ),
				"c2" => array( "'", "'" )
			),
			array(
				"c1" => array( "（", "（" ),
				"c2" => array( "(", "(" )
			),
			array(
				"c1" => array( "）", "）" ),
				"c2" => array( ")", ")" )
			),
			array(
				"c1" => array( "＊", "＊" ),
				"c2" => array( "*", "*" )
			),
			array(
				"c1" => array( "＋", "＋" ),
				"c2" => array( "+", "+" )
			),
			array(
				"c1" => array( "，", "，" ),
				"c2" => array( ",", "," )
			),
			array(
				"c1" => array( "−", "−" ),
				"c2" => array( "-", "-" )
			),
			array(
				"c1" => array( "ー", "ー" ),
				"c2" => array( "-", "-" )
			),
			array(
				"c1" => array( "．", "．" ),
				"c2" => array( ".", "." )
			),
			array(
				"c1" => array( "／", "／" ),
				"c2" => array( "/", "/" )
			),
			array(
				"c1" => array( "：", "：" ),
				"c2" => array( ":", ":" )
			),
			array(
				"c1" => array( "；", "；" ),
				"c2" => array( ";", ";" )
			),
			array(
				"c1" => array( "＜", "＜" ),
				"c2" => array( "<", "<" )
			),
			array(
				"c1" => array( "＝", "＝" ),
				"c2" => array( "=", "=" )
			),
			array(
				"c1" => array( "＞", "＞" ),
				"c2" => array( ">", ">" )
			),
			array(
				"c1" => array( "？", "？" ),
				"c2" => array( "?", "?" )
			),
			array(
				"c1" => array( "＠", "＠" ),
				"c2" => array( "@", "@" )
			),
			array(
				"c1" => array( "［", "［" ),
				"c2" => array( "[", "[" )
			),
			array(
				"c1" => array( "＼", "＼" ),
				"c2" => array( "\\", "\\" )
			),
			array(
				"c1" => array( "］", "］" ),
				"c2" => array( "]", "]" )
			),
			array(
				"c1" => array( "＾", "＾" ),
				"c2" => array( "^", "^" )
			),
			array(
				"c1" => array( "＿", "＿" ),
				"c2" => array( "_", "_" )
			),
			array(
				"c1" => array( "‘", "‘" ),
				"c2" => array( "`", "`" )
			),
			array(
				"c1" => array( "｛", "｛" ),
				"c2" => array( "{", "{" )
			),
			array(
				"c1" => array( "｜", "｜" ),
				"c2" => array( "|", "|" )
			),
			array(
				"c1" => array( "｝", "｝" ),
				"c2" => array( "}", "}" )
			),
			array(
				"c1" => array( "〜", "〜" ),
				"c2" => array( "~", "~" )
			),
			array(
				"c1" => array( "　", "　" ),
				"c2" => array( " ", " " )
			)
		);

		return self::tr( $data, $ctbl, $isz2h );
	}
}
?>
