<?php
/**
 * Mime header class
 *
 * This file defines YggMimeHeader class.
 * It includes methods to support converting to mime header.
 *
 * @package YggDore.Base.Locale
 * @author YggDore Co.,Ltd.
 */


/**
 * Require config file
 */
require_once( "YggDore/Base/Config.php" );
/**
 * Require YggStringIconv class
 */
require_once( "YggDore/Base/Locale/YggStringIconv.php" );


/**
 * YggDore mime header class
 *
 * This class has methods to support converting to mime header.
 *
 * @package YggDore.Base.Locale
 */
class YggMimeHeader {
	/**
	 * Calculate data block count when encodes to mime
	 *
	 * This method calculates data block count. One block of mime data part is 4 bytes.<br />
	 * $denclen is specified byte scale of data.<br />
	 * $linesz is byte scale of one line.
	 *
	 * <code>
	 * $rts = YggMimeHeader::calcBlock( strlen( "ISO-2022-JP" ) );
	 * </code>
	 *
	 * @param integer $denclen Byte scale of data
	 * @param integer $linesz Byte scale of one line
	 * @return integer Block count
	 */
	public static function calcBlock( $denclen, $linesz = 76 )
	{
		if( $denclen <= 0 ){
			throw new RangeException;
		}
		elseif( $linesz <= 0 ){
			throw new RangeException;
		}

		// 7 is equal with byte scale of [=? ?B? ?=] characters
		// 4 is equal with byte scale of one mime block size
		$blk = (int)( ( $linesz - 7 - $denclen ) / 4 );
		if( $blk < 0 ){
			return 0;
		}

		return $blk;
	}


	/**
	 * Create MIME encode line
	 *
	 * This method creates "=?$denc?B?to_mime_$denc?=" format string from $data.<br />
	 * $didx is starting point converted of $data. It is scale of character, isn't byte.
	 * This method supports to convert multibyte character string.<br />
	 * if $data is specified empty string or converting $data failed,
	 * Array that has elements dlen is 0 and MIME is empty string is returned.<br />
	 * <br />
	 * <code>
	 * $data = YggMimeHeader::convLine( "hogehoge" );
	 * </code>
	 *
	 * @param string $data Target data
	 * @param integer $didx Index of $data
	 * @param integer $linesz One line size
	 * @param string $senc String encode of $data<br />
	 * Internal encoding is used if it is empty.
	 * @param string $denc Destination encoding<br />
	 * Internal encoding is used if it is empty.
	 * @return array Result<br />
	 * (It has the following elements)<br />
	 * dlen : (integer) Number of used character that is used to create one line mime header<br />
	 * mime : (string) Mime data
	 */
	public static function convLine( $data, $didx = 0, $linesz = 76, $senc = "", $denc = "" )
	{
		if( $didx < 0 ){
			throw new RangeException;
		}
		elseif( $linesz <= 0 ){
			throw new RangeException;
		}

		$ienc = iconv_get_encoding( "internal_encoding" );
		if( $senc == "" ){
			$senc = $ienc;
		}
		if( $denc == "" ){
			$denc = $ienc;
		}

		$bsz = self::calcBlock( strlen( $denc ), $linesz );

		$erts = array( "dlen" => 0, "mime" => "" );

		$dlen = iconv_strlen( $data, $senc );
		if( $dlen <= 0 ){
			return $erts;
		}

		if( $data == "" || $didx >= $dlen ){
			return $erts;
		}

		$ldsz = $bsz * 4;

		$ldat = "";
		for( $i = 0; $i < $dlen - $didx; $i++ ){
			$chr = iconv_substr( $data, $i + $didx, 1, $senc );
			if( $chr == "" ){
				throw new RuntimeException;
			}

			$estr = YggStringIconv::conv( $ldat . $chr, $senc, $denc );
			if( $estr == "" ){
				throw new RuntimeException;
			}

			$dsz = (int)( ( strlen( $estr ) + 2 ) / 3 ) * 4; 
			
			if( $dsz > $ldsz ){
				break;
			} 

			$ldat .= $chr;
		} 
		if( $ldat == "" ){
			return $erts;
		}
		
		$ldat = YggStringIconv::conv( $ldat, $senc, $denc );
		if( $ldat == "" ){
			throw new RuntimeException;
		} 
		
		$ldat = base64_encode( $ldat );
		if( $ldat == "" ){
			throw new RuntimeException;
		}

		return array(
			"dlen" => $i,
			"mime" => "=?" . $denc . "?B?" . $ldat . "?="
		);
	}


	/**
	 * Create MIME encode line
	 *
	 * This method converts $data to "=?$denc?B?to_mime_$denc?=" format.<br />
	 * $fhd is added to first line and position in $data.<br />
	 * $eolm is specified new line character mode.<br />
	 * $linesz is line byte size. Its size is specified scale of bytes.
	 * It includes $fhd and new line character.
	 * If size of $fhd is bigger than $linesz, error occurs.
	 *
	 * <code>
	 * $data = YggMimeHeader::conv(
	 *     "Title", "Subject: ", YMH_EOLM_LF, 76, null, "ISO-2022-JP" 
	 * );
	 * $data = YggMimeHeader::conv( "hogehoge" );
	 * </code>
	 *
	 * @param string $data Target data
	 * @param string $fhd Head data
	 * @param integer $eolm New line character mode<br />
	 * YggString::EOL : YGG_EOL in Config.php<br />
	 * YggString::CR : CR character<br />
	 * YggString::LF : LF character<br />
	 * YggString::CRLF : CRLF characters
	 * @param integer $linesz Byte size of one line
	 * @param string $senc String encode of $data<br />
	 * Internal encoding is used if it is empty.
	 * @param string $denc Destination encoding<br />
	 * Internal encoding is used if it is empty.
	 * @return string
	 */
	public static function conv(
		$data,
		$fhd = "",
		$eolm = YggString::EOL,
		$linesz = 76,
		$senc = "",
		$denc = ""
	)
	{
		if( $eolm < YggString::EOL || $eolm > YggString::CRLF ){
			throw new UnexpectedValueException;
		}

		$ienc = iconv_get_encoding( "internal_encoding" );
		if( $senc == "" ){
			$senc = $ienc;
		}
		if( $denc == "" ){
			$denc = $ienc;
		}

		$rc = YggString::eol( YGG_EOL, -1, $eolm );

		$linesz -= strlen( $rc );
		if( $linesz <= 0 ){
			throw new UnexpectedValueException;
		}

		$fhdl = strlen( $fhd );
		if( $fhdl > $linesz ){
			throw new UnexpectedValueException;
		}

		if( $fhd != "" ){
			$fhd = YggString::eol( $fhd, -1 );
			if( strpos( $fhd, YGG_EOL ) !== false ){
				throw new UnexpectedValueException;
			}
		}

		$dlen = iconv_strlen( $data, $senc );
		if( $dlen <= 0 ){
			return $fhd;
		}

		// Convert first line
		$rts = self::convLine( $data, 0, $linesz - $fhdl, $senc, $denc );
		if( $rts['dlen'] <= 0 ){
			throw new UnexpectedValueException;
		}

		$menc = $fhd . $rts['mime'];
		$rpos = $rts['dlen'];

		// Convert second line and more
		while( $rpos < $dlen ){
			// 1 of [linesz-1] is equal to space character
			// Second and more line begins space character
			$rts = self::convLine( $data, $rpos, $linesz - 1, $senc, $denc );
			if( $rts['dlen'] <= 0 ){
				throw new RuntimeException;
			}

			$menc .= $rc . " " . $rts['mime'];

			$rpos += $rts['dlen'];
		}

		return $menc;
	}
}
?>
