<?php
/**
 * String class
 *
 * This file defines YggStringIconv class.
 * It includes methods to operate multi bytes string.
 *
 * @package YggDore.Base.Locale
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );
/**
 * Require YggInt class
 */
require_once( "YggDore/Base/YggInt.php" );



/**
 * String class
 *
 * This class has methods to operate multi bytes string.
 *
 * @package YggDore.Base.Locale
 */
class YggStringIconv {
	/**
	 * Create random charctors string
	 * 
	 * This method creates random digit and alphabet, $addstr characters string.<br />
	 * This method is slower than YggString::random method,
	 * but it is supported that $addstr is specified multi bytes character.
	 *
	 * <code>
	 * $rts = YggStringIconv::random( 15, "%-_" );
	 * </code>
	 *
	 * @param integer $len String length
	 * @param string $addstr Random addition string
	 * @param string $enc String encode of $data<br />
	 * Internal encoding is used if it is empty.
	 * @return string
	 */
	public static function random( $len, $addstr = "", $enc = "" )
	{
		if( $len <= 0 ){
			throw new RangeException;
		}

		$ienc = iconv_get_encoding( "internal_encoding" );
		if( $enc == "" ){
			$enc = $ienc;
		}

		$pstbl = "0123456789" .
		         "abcdefghijkmnopqrstuvwxyz" .
				 "ABCDEFGHJKLMNPQRSTUVWXYZ";

		$pstbl .= $addstr;

		$slen = iconv_strlen( $pstbl, $enc );
		if( $slen <= 0 ){
			throw new RuntimeException;
		}

		$rts = "";
		for( $i = 0; $i < $len; $i++ ){
			$idx = mt_rand( 0, $slen - 1 );

			$rts .= iconv_substr( $pstbl, $idx, 1, $enc );
		}

		return $rts;
	}


	/**
	 * Convert to stable string
	 * 
	 * This method converts type of $data to string.<br />
	 * If string length of $data is more than $limit, it fixes to $limit.
	 * If NULL code is found in it, string It is cut before NULL.
	 * This function defends NULL byte attach.<br />
	 * $data is converted from $senc to $enc encode.<br />
	 * "//TRANSLIT" and "//IGNORE" options of iconv function is "not" able to
	 * be used.<br />
	 * This function doesn't send warning message when type of $data is anything.
	 *
	 * <code>
	 * $data = "150\0test";
	 * $rts  = YggStringIconv::stable( $data, "SJIS", 1024 );
	 * (Result)
	 * $rts => "150"
	 * </code>
	 *
	 * @param mixed $data Target string
	 * @param string $senc String encode of $data<br />
	 * Internal encoding is used if it is empty.
	 * @param string $denc Destination string encode<br />
	 * Internal encoding is used if it is empty.
	 * @param integer $limit Limit string length
	 * @return string
	 */
	public static function stable( $data, $senc = "", $denc = "", $limit = YGG_YSSLIMIT )
	{
		if( $limit < 0 ){
			throw new UnexpectedValueException;
		}

		$ienc = iconv_get_encoding( "internal_encoding" );
		if( $senc == "" ){
			$senc = $ienc;
		}
		if( $denc == "" ){
			$denc = $ienc;
		}

		$tiflg = "//TRANSLIT//IGNORE";

		$data = YggString::stable( $data, $limit );
		$data = iconv( $senc, $denc . $tiflg, $data );
		if( !is_string($data) ){
			return "";
		}
		
		$data = YggString::stable( $data, $limit );
		$data = iconv( $denc, $denc . $tiflg, $data );
		if( !is_string($data) ){
			return "";
		}

		return $data;
	}


	/**
	 * Convert to stable strings in array
	 * 
	 * This method converts type of each elements specified keys by $keyl in $data to string.<br />
	 * If string length of each elements of $data is more than $limit, it fixes to $limit.
	 * If NULL code is found in each elements, the string of the elements is cut before NULL.
	 * This function defends NULL byte attach.<br />
	 * $data elements are converted from $senc to $enc encode.
	 * This function doesn't send warning message when type of elements of $data is anything.
	 *
	 * <code>
	 * $data = array( "name" => "Ito\0yan", "test" => 15 );
	 * $rts  = YggStringIconv::stableArray(
	 *     $data, array( "name" ), "JIS", 1024
	 * );
	 * (Result)
	 * $rts => array( "name" => "Ito" )
	 * </code>
	 *
	 * @param array $data Target string array
	 * @param array $keyl Target key list
	 * @param string $senc String encode of $data elements<br />
	 * Internal encoding is used if it is empty.
	 * @param string $denc Destination string encode<br />
	 * Internal encoding is used if it is empty.
	 * @param integer $limit Limit string length
	 * @return array
	 */
	public static function stableArray( $data, $keyl, $senc = "", $denc = "", $limit = YGG_YSSLIMIT )
	{
		$data = YggSafe::safeArray( $data );
		$data = YggArray::extract( $data, $keyl );

		foreach( $keyl as $key ){
			$data[$key] = self::stable( $data[$key], $senc, $denc, $limit );
		}

		return $data;
	}


	/**
	 * Convert string encoding
	 *
	 * This method converts string encoding from $senc to $denc in $data.<br />
	 * If $denc or $senc is empty string, internal encoding is used.
	 * "//TRANSLIT" and "//IGNORE" options of iconv function is able to
	 * be used.<br />
	 * UnexpectedValueException is thrown if error occurs.
	 *
	 * <code>
	 * $data = "  Hello1!Good1!  ";
	 * $data = YggStringIconv::conv( $data, null, "UTF-8" );
	 * </code>
	 *
	 * @param string $data Taget data
	 * @param string $senc Source string encoding
	 * @param string $denc Destination string encoding
	 * @return string
	 */
	public static function conv( $data, $senc = "", $denc = "" )
	{
		if( $data == "" ){
			return "";
		}

		$ienc = iconv_get_encoding( "internal_encoding" );

		// If source or destination string encoding is empty,
		// Use them internal encoding
		if( $senc == "" ){
			$senc = $ienc;
		}
		if( $denc == "" ){
			$denc = $ienc;
		}

		$rts = iconv( $senc, $denc, $data );
		if( !is_string($rts) ){
			throw new UnexpectedValueException;
		}

		return $rts;
	}


	/**
	 * Convert code
	 * 
	 * This method converts $data string by $ctbl converting information.<br />
	 * Characters in $data are converted from "c1" to "c2" in $ctbl if $c12 is specified TRUE,
	 * otherwize from "c2" to "c1".<br />
	 * $enc is specified encoding of $data and $ctbl.
	 *
	 * <code>
	 * $ctbl = array(
	 *     array(
	 *         "c1" => array( "a", "c" ),
	 *         "c2" => array( "c", "a" )
	 *     )
	 * );
	 * $rts = YggStringIconv::tr( "aig", $ctbl );
	 * // rts = "cig"
	 * </code>
	 *
	 * @param string $data Target data
	 * @param array $ctbl Converting table<br />
	 * (array in array haves elements as follows)<br />
	 * c1 : Convert base 1
	 * c2 : Convert base 2
	 * @param boolean Convert from c1 to c2 if $isr2 is specified TRUE,
	 * otherwise from c2 to c1.
	 * @param string $enc String encode of $data<br />
	 * Internal encoding is used if it is empty.
	 * @return string
	 */
	public static function tr( $data, $ctbl, $c12 = true, $enc = "" )
	{
		if( count($ctbl) <= 0 ){
			throw new UnexpectedValueException;
		}

		$ienc = iconv_get_encoding( "internal_encoding" );
		if( $enc == "" ){
			$enc = $ienc;
		}

		$len = iconv_strlen( $data, $enc );
		if( $len <= 0 ){
			return "";
		}

		// Get converting key name of source and destination
		$sk = "c1";
		$dk = "c2";
		if( !$c12 ){
			$sk = "c2";
			$dk = "c1";
		}

		// Devide each elements of $ctbl per a byte and
		// find the reminder of source and destination
		$btbl = array( "c1", "c2" );
		for( $i = 0; $i < count($ctbl); $i++ ){
			foreach( $btbl as $bkey ){
				$ctrec =& $ctbl[$i][$bkey];
				$ctrec[0] = unpack( "C*", $ctrec[0] );
				$ctrec[1] = unpack( "C*", $ctrec[1] );
				$ctrec[2] = self::_trbm( $ctrec[1], $ctrec[0] );
				$ctrec[3] = $ctrec[2];

				// When reversing data is specified, reverses them
				if( $ctrec[2] < 0 ){
					$w        = $ctrec[0];
					$ctrec[0] = $ctrec[1];
					$ctrec[1] = $w;
					$ctrec[2] = abs( $ctrec[2] );
					$ctrec[3] = 0;
				}
			}
			if( $ctbl[$i]['c1'][2] != $ctbl[$i]['c2'][2] ){
				throw new UnexpectedValueException;
			}
		}

		$rts = "";
		for( $i = 0; $i < $len; $i++ ){
			$schr = iconv_substr( $data, $i, 1, $enc );
			$upsc = unpack( "C*", $schr );

			// Check whether the character code is range of replacing
			$dchr = "";
			foreach( $ctbl as $cr ){
				$upscm = self::_trbm( $upsc, $cr[$sk][0] );
				if( $upscm < 0 || $upscm > $cr[$sk][2] ){
					continue;
				}

				// Calclate destination point supported reversal table
				$upscm = abs( $upscm - abs( $cr[$dk][3] - $cr[$sk][3] ) );

				$dchr  = self::_trbpc( $cr[$dk][0], $upscm );
				break;
			}
			if( $dchr == "" ){
				$dchr = $schr;
			}

			$rts .= $dchr;
		}

		return $rts;
	}


	/**
	 * Calculate minus to two characters
	 * 
	 * This method calculates that $uv1 minors $uv2.<br />
	 * $uv1 and $uv2 are unpacked character data.
	 *
	 * @param array $uv1 Unpacked value 1
	 * @param array $uv2 Unpacked value 2
	 * @return integer
	 */
	private static function _trbm( $uv1, $uv2 )
	{
		// Support maximum value
		$rm = YGG_INTMAX - 256;

		// Make figure of two value even
		$fp  =& $uv1;
		$uvl =  count( $uv2 );
		if( count( $uv1 ) > count( $uv2 ) ){
			$fp  =& $uv2;
			$uvl =  count( $uv1 );
		}
		for( $i = 1; $i <= $uvl; $i++ ){
			if( !isset( $fp[$i] ) ){
				$fp[$i] = 0;
			}
		}

		// $uv1 minor $uv2
		$rts = 0;
		$sb  = 1;
		for( $i = 1; $i <= $uvl; $i++ ){
			$rts *= $sb;
			$sb   = 256;

			$rts += $uv1[$i] - $uv2[$i];
		}
		if( !YggInt::is( $rts ) || $rts >= $rm ){
			throw new OverflowException;
		}

		return $rts;
	}


	/**
	 * Calculate plus to two characters and get character of the result
	 * 
	 * This method calculates that $v1 pluses $v2 and returns character of the value.<br />
	 * $v1 is unpacked character data.<br />
	 * $sub is specified subtraction from $v1.
	 *
	 * @param array $uv Unpacked value
	 * @param integer $sub Subtraction from $v1.
	 * @return string
	 */
	private static function _trbpc( $uv, $sub )
	{
		// Maximum value of char
		$cm = 256;

		if( $sub < 0 ){
			throw UnexpectedValueException;
		}

		// Plus $uv and $sub
		$rlt = array();
		for( $i = count( $uv ); $i >= 1; $i-- ){
			$w = $uv[$i] + $sub;

			$rlt[] = $w % $cm;
			$sub   = (int)( $w / $cm );
		}
		while( $sub > 0 ){
			$rlt[] = $sub % $cm;
			$sub   = (int)( $w / $cm );
		}

		// Create character
		// Elements of result array are reversed
		// because result is reversal array now, 
		return call_user_func_array(
			"pack", array_merge( array( "C*" ), array_reverse( $rlt ) )
		);
	}
}
?>
