<?php
/**
 * String class
 *
 * This file defines YggStringUTF8 class.
 * It includes methods to convert string to utf-8 encoding.
 *
 * @package YggDore.Base.Locale
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );
/**
 * Require YggStringIconv class
 */
require_once( "YggDore/Base/Locale/YggStringIconv.php" );



/**
 * String class
 *
 * This class has methods to convert string to utf-8 encoding.
 *
 * @package YggDore.Base.Locale
 */
class YggStringUTF8 extends YggString {
	/**
	 * Replace from between % character to word
	 *
	 * This method searches point between % character in $src
	 * and replaces from %name% to $chtbl[name] in it.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $src = "%name1% %name2%";
	 * $chtbl = array(
	 *     "name1" => "fujiyama",
	 *     "name2" => "geisya"
	 * );
	 * $dst = YggStringUTF8::embed($src, $chtbl);
	 * (Result)
	 * $dst => fujiyama geisya
	 * </code>
	 *
	 * @param string $src Target string
	 * @param array $chtbl Replace list<br />
	 * Its key is used to look up the word in $data and replace to value.
	 * @return string
	 */
	public static function embed( $src, $chtbl )
	{
		$chgtbl = array( "%%" => "%" );

		if( count($chtbl) <= 0 ){
			throw new UnexpectedValueException;
		}

		if( $src == "" ){
			return "";
		}

		foreach( $chtbl as $key => $val ){
			if( strlen($key) <= 0 ){
				throw new UnexpectedValueException;
			}

			$chgtbl['%'.$key.'%'] = $val;
		}
		return (string)preg_replace(
			'/(%[^%]*%)/eu',
			'isset($chgtbl[\'$1\']) ? $chgtbl[\'$1\'] : ""',
			$src
		);
	}


	/**
	 * Create random characters string
	 * 
	 * This method creates random digit and alphabet, $addstr characters string.<br />
	 * Its processing is slower than YggString::random method because iconv function is used.
	 * But you can specify $addstr that has zenkaku characters.
	 *
	 * <code>
	 * $rts = YggStringUTF8::random( 15, "%-_" );
	 * </code>
	 *
	 * @param integer $len String length
	 * @param string $addstr Random addition string
	 * @return string
	 */
	public static function random( $len, $addstr = "" )
	{
		return YggStringIconv::random( $len, $addstr, "UTF-8" );
	}


	/**
	 * Convert to stable string
	 * 
	 * This method converts type of $data to string.<br />
	 * If string length of $data is more than $limit, it fixes to $limit.
	 * If NULL code is found in it, it is cut before NULL.
	 * This function defends NULL byte attach.<br />
	 * $data is converted from $enc to UTF-8 encode.<br />
	 * This function doesn't send warning message when type of $data is anything.
	 *
	 * <code>
	 * $data = "150\0test";
	 * $rts  = YggStringUTF8::stable( $data, "SJIS", 1024 );
	 * (Result)
	 * $rts => "150"
	 * </code>
	 *
	 * @param mixed $data Target string
	 * @param string $enc String encode of $data<br />
	 * Internal encoding is used if it is empty.
	 * @param integer $limit Limit string length
	 * @return string
	 */
	public static function stable( $data, $enc = "", $limit = YGG_YSSLIMIT )
	{
		return YggStringIconv::stable( $data, $enc, "UTF-8", $limit );
	}


	/**
	 * Convert to stable strings in array
	 * 
	 * This method converts type of each elements specified keys by $keyl in $data to string.<br />
	 * If string length of each elements of $data is more than $limit, it fixes to $limit.
	 * If NULL code is found in each elements, the string of the elements is cut before NULL.
	 * This function defends NULL byte attach.<br />
	 * $data elements are converted from $enc to UTF-8.<br />
	 * This function doesn't send warning message when type of elements of $data is anything.
	 *
	 * <code>
	 * $data = array( "name" => "Ito\0yan", "test" => 15 );
	 * $rts  = YggStringUTF8::stableArray(
	 *     $data, array( "name" ), "JIS", 1024
	 * );
	 * (Result)
	 * $rts => array( "name" => "Ito" )
	 * </code>
	 *
	 * @param array $data Target string array
	 * @param array $keyl Target key list
	 * @param string $enc String encode of $data elements<br />
	 * Internal encoding is used if it is empty.
	 * @param integer $limit Limit string length
	 * @return array
	 */
	public static function stableArray( $data, $keyl, $enc = "", $limit = YGG_YSSLIMIT )
	{
		return YggStringIconv::stableArray( $data, $keyl, $enc, "UTF-8", $limit );
	}


	/**
	 * Convert code
	 * 
	 * Convert $data from $ctbl.<br />
	 * Characters in $data are converted from "c1" to "c2" in $ctbl if $c12 is specified TRUE,
	 * otherwize from "c2" to "c1".<br />
	 * $enc is specified encoding of $data and $ctbl.
	 *
	 * <code>
	 * $ctbl = array(
	 *     array(
	 *         "c1" => array( "a", "c" ),
	 *         "c2" => array( "b", "d" )
	 *     )
	 * );
	 * $rts = YggStringUTF8::tr( "aig", $ctbl );
	 * // rts = "big"
	 * </code>
	 *
	 * @param string $data Target data
	 * @param array $ctbl Converting table<br />
	 * (array in array haves elements as follows)<br />
	 * c1 : Convert base 1
	 * c2 : Convert base 2
	 * @param boolean Convert from c1 to c2 if $isr2 is specified TRUE,
	 * otherwise from c2 to c1.
	 * @return string
	 */
	public function tr( $data, $ctbl, $c12 = true )
	{
		return YggStringIconv::tr( $data, $ctbl, $c12, "UTF-8" );
	}
}
?>
