<?php
/**
 * Date and time class
 *
 * This file defines YggDateTime class.
 * It includes methods to compare date method and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Date and time class
 *
 * This class has methods to compare date method and so on.
 *
 * <code>
 * $dtobj = new YggDateTime( 2005, 5, 28, 17, 39, 30 );
 * $dtobj->getUnixTimeStamp();
 * </code>
 *
 * @package YggDore.Base
 */
class YggDateTime {
	/**
	 * Compare status : Past
	 *
	 * This value is used as return value of YggDateTime::compare method.
	 * It shows past date and time.
	 */
	const PAST = -1;
	/**
	 * Compare status : Same
	 *
	 * This value is used as return value of YggDateTime::compare method.
	 * It shows same date and time.
	 */
	const SAME = 0;
	/**
	 * Compare status : Future
	 *
	 * This value is used as return value of YggDateTime::compare method.
	 * It shows future date and time.
	 */
	const FUTURE = 1;


	/**
	 * Unix time stamp
	 */
	private $_ut;


	/**
	 * Constructor
	 *
	 * The instance is initialized.<br />
	 * Each arguments are specified initial value.
	 *
	 * @param integer $year Year
	 * @param integer $month Month
	 * @param integer $day Day
	 * @param integer $hour Hour
	 * @param integer $minute Minute
	 * @param integer $second Second
	 * @param string_or_null $tz Time zone of date and time arguments<br />
	 * ( If the argument is set NULL, time zone is used current setting )
	 */
	public function __construct(
		$year = 0, $month = 0, $day = 0,
		$hour = 0, $minute = 0, $second = 0,
		$tz = ""
	)
	{
		$this->clear();

		// Call set method when arguments of date and time aren't null
		// because arguments of set method don't allow to set 0.
		$vtbl = array( &$year, &$month, &$day, &$hour, &$minute, &$second );
		$ise  = true;
		foreach( $vtbl as $val ){
			if( $val != 0 ){
				$ise = false;
				break;
			}
		}
		if( $ise ){
			return;
		}

		$this->set( $year, $month, $day, $hour, $minute, $second, $tz );
	}


	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_ut = null;
	}


	/**
	 * Set date and time
	 *
	 * Set date and time in the instance.<br />
	 * This method throws RangeException or UnexpectedValueException,
	 * RuntimeException when error occurs.
	 *
	 * @param integer $year Year
	 * @param integer $month Month
	 * @param integer $day Day
	 * @param integer $hour Hour
	 * @param integer $minute Minute
	 * @param integer $second Second
	 * @param string_or_null $tz Time zone of date and time arguments<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 */
	public function set(
		$year, $month, $day,
		$hour = 0, $minute = 0, $second = 0,
		$tz = ""
	)
	{
		if( $year < 1970 ){
			throw new RangeException;
		}
		elseif( $month < 1 || $month > 12 ){
			throw new RangeException;
		}
		elseif( $day < 1 || $day > 31 ){
			throw new RangeException;
		}
		elseif( $hour < 0 || $hour > 23 ){
			throw new RangeException;
		}
		elseif( $minute < 0 || $minute > 59 ){
			throw new RangeException;
		}
		elseif( $second < 0 || $second > 59 ){
			throw new RangeException;
		}
		elseif( !YggDateTime::checkDate( $year, $month, $day ) ){
			throw new UnexpectedValueException;
		}

		// The function calls before calling bundled date and time functions
		// because they use timezone and calculate.
		$ctz = self::_setTimeZone( $tz );

		$ut = mktime( $hour, $minute, $second, $month, $day, $year );
		if( !is_int($ut) || $ut < 0 ){
			self::_setTimeZone( $ctz );
			throw new RuntimeException;
		}
		$this->_ut = $ut;

		self::_setTimeZone( $ctz );
	}


	/**
	 * Set unix time stamp
	 *
	 * Set unix time stamp in the instance.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param integer $ut Timestamp
	 */
	public function setUnixTimeStamp( $ut )
	{
		if( $ut < 0 ){
			throw new UnexpectedValueException;
		}

		$this->_ut = $ut;
	}


	/**
	 * Set string format date and time
	 *
	 * This method calls strtotime and $datetime converts to time numeric.<br />
	 * This method throws RuntimeException or UnexpectedValueException
	 * when error occurs.
	 *
	 * @param string $datetime Date and time string
	 * @param string_or_null $tz Time zone of date and time arguments<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 */
	public function setString( $datetime, $tz = "" )
	{
		if( $datetime == "" ){
			throw new UnexpectedValueException;
		}

		$ctz = self::_setTimeZone( $tz );

		$ut = strtotime( $datetime );
		if( $ut < 0 ){
			throw new RuntimeException;
		}

		self::_setTimeZone( $ctz );

		$this->_ut = $ut;
	}


	/**
	 * Get year
	 *
	 * This method returns value of the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return integer
	 */
	public function getYear( $tz = "" )
	{
		return (int)$this->getString( "Y", $tz );
	}


	/**
	 * Get month
	 *
	 * This method returns value of the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return integer
	 */
	public function getMonth( $tz = "" )
	{
		return (int)$this->getString( "m", $tz );
	}


	/**
	 * Get day
	 *
	 * This method returns value of the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return integer
	 */
	public function getDay( $tz = "" )
	{
		return (int)$this->getString( "d", $tz );
	}


	/**
	 * Get hour
	 *
	 * This method returns value of the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return integer
	 */
	public function getHour( $tz = "" )
	{
		return (int)$this->getString( "H", $tz );
	}


	/**
	 * Get minute
	 *
	 * This method returns value of the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return integer
	 */
	public function getMinute( $tz = "" )
	{
		return (int)$this->getString( "i", $tz );
	}


	/**
	 * Get second
	 *
	 * This method returns value of the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return integer
	 */
	public function getSecond( $tz = "" )
	{
		return (int)$this->getString( "s", $tz );
	}


	/**
	 * Get string format date and time
	 *
	 * This method returns value of the instance.<br />
	 * This method calls date funtion and converts instance value to string.
	 * $format is specified argument same as date function.<br />
	 * This method throws RuntimeException or BadMethodCallException,
	 * UnexpectedValueException when error occurs.
	 *
	 * @param string $format Format string
	 * @param string_or_null $tz Time zone of the instance<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return string
	 */
	public function getString( $format, $tz = "" )
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		if( $format == "" ){
			throw new UnexpectedValueException;
		}

		$ctz = self::_setTimeZone( $tz );

		$rts = date( $format, $this->_ut );
		if( !is_string( $rts ) ){
			self::_setTimeZone( $ctz );
			throw new RuntimeException;
		}

		self::_setTimeZone( $ctz );

		return $rts;
	}


	/**
	 * Get unix time stamp
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer
	 */
	public function getUnixTimeStamp()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_ut;
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_ut !== null ){
			return true;
		}

		return false;
	}


	/**
	 * Compare date and time
	 *
	 * This method compares the instance and $dtobj.
	 *
	 * @param YggDateTime $dtobj Target object
	 * @return integer Result<br />
	 * YggDateTime::PAST : this instance is past date and time than $dtobj.<br />
	 * YggDateTime::SAME : this instance is same date and time than $dtobj.<br />
	 * YggDateTime::FUTURE : this instance is future date and time than $dtobj.
	 */
	public function compare( $dtobj )
	{
		$tut = $this->getUnixTimeStamp();
		$aut = $dtobj->getUnixTimeStamp();

		if( $tut > $aut ){
			return self::FUTURE;
		}
		elseif( $tut < $aut ){
			return self::PAST;
		}

		return self::SAME;
	}


	/**
	 * Check whether date is valid
	 *
	 * This method checks whether date of argument is valid.
	 *
	 * @param integer $year Year
	 * @param integer $month Month
	 * @param integer $day Day
	 * @return boolean Returns TRUE If the date given is valid,
	 * otherwise returns FALSE.
	 */
	public static function checkDate( $year, $month, $day )
	{
		return checkdate( $month, $day, $year );
	}


	/**
	 * Create and get now
	 *
	 * This method creates a instance of this class and
	 * set current date and time.
	 *
	 * @return YggDateTime
	 */
	public static function createNow()
	{
		$nowt = time();

		$dt = new YggDateTime;
		$dt->setUnixTimeStamp( $nowt );

		return $dt;
	}


	/**
	 * Create and get today
	 *
	 * This method creates a instance of this class and
	 * set current date. Its time area (hour, minute, second) are set 0.
	 *
	 * @param string_or_null $tz Time zone of date and time arguments<br />
	 * ( If this value is set NULL, time zone is used current setting )
	 * @return YggDateTime
	 */
	public static function createToday( $tz = "" )
	{
		$dt = self::createNow();

		$dt->set(
			$dt->getYear( $tz ),
			$dt->getMonth( $tz ),
			$dt->getDay( $tz ),
			0,
			0,
			0,
			$tz
		);

		return $dt;
	}


	/**
	 * Set time zone
	 *
	 * Set time zone used in "this php program".
	 * (not this instance)<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param string $tz Time zone
	 * @return string Previous time zone
	 */
	private static function _setTimeZone( $tz )
	{
		if( $tz == "" ){
			return "";
		}

		$ctz = date_default_timezone_get();
		$rts = date_default_timezone_set( $tz );
		if( !$rts ){
			throw new UnexpectedValueException;
		}

		return $ctz;
	}
}
?>
