<?php
/**
 * Delegate class
 *
 * This file defines YggDelegate class.
 * It includes methods to manage delegate.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Delegate class
 *
 * This class has methods to manage delegate.
 *
 * <code>
 * $dobj = new YggDelegate;
 * $dobj->set( array( &$handlerobj, "delegateClick" ) );
 * $dobj->exec( array( "test", &$refdata ) );
 * </code>
 *
 * @package YggDore.Base
 */
class YggDelegate {
	/**
	 * Delegate status : No set
	 *
	 * This value is used as return value of YggDelegate::getStatus method.
	 * It shows that delegate don't set.
	 */
	const NONE = 0;
	/**
	 * Delegate status : Callable
	 *
	 * This value is used as return value of YggDelegate::getStatus method.
	 * It shows that delegate can call.
	 */
	const CALLABLE = 1;
	/**
	 * Delegate status : Illegal
	 *
	 * This value is used as return value of YggDelegate::getStatus method.
	 * It shows that delegate can't call.
	 */
	const ILLEGAL  = 2;


	/**
	 * Delegate callback
	 */
	private $_call;

	
	/**
	 * Constructor
	 *
	 * The instance is initialized.<br />
	 * Each arguments are specified initial value.
	 *
	 * @param array $callback Delegate callback
	 */
	public function __construct( $callback = null )
	{
		$this->clear();
		if( $callback == null ){
			return;
		}
		$this->set( $callback );
	}

	
	/**
	 * Set delegate callback
	 *
	 * Delegate callback is set to the instance.
	 * 
	 * The argument can be specified type of callback which
	 * is function or object method.<br />
	 * It is same as argument of is_callable and call_user_func_array function.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $callback Callback
	 */
	public function set( $callback )
	{
		if( !is_callable( $callback ) ){
			throw new UnexpectedValueException;
		}
		
		$this->_call = $callback;
	}


	/**
	 * Get delegate callback
     *
	 * This method returns delegate callback.<br />
	 * If this method call before callback setting,
	 * BadMethodCallException is thrown.
	 *
	 * @return array Callback
     */
	public function get()
	{
		if( !is_callable( $this->_call ) ){
			throw new BadMethodCallException;
		}
		
		return $this->_call;
	}


	/**
	 * Clear
     *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_call = null;
	}


	/**
	 * Get callback status
	 *
	 * This method returns whether the callback is callable.
     *
	 * @return integer Delegate status<br />
	 * YggDelegate::NONE No setting<br />
	 * YggDelegate::CALLABLE Callable<br />
	 * YggDelegate::ILLEGAL No callable
	 */
	public function getStatus()
	{
		if( !isset( $this->_call ) ){
			return self::NONE;
		}
		elseif( !is_callable( $this->_call ) ){
			return self::ILLEGAL;
		}

		return self::CALLABLE;
	}


	/**
	 * Execute callback
     *
	 * Execute callback in the instance.<br />
	 * It is called as follows.<br />
	 * <br />
	 * <code>
	 * mixed callback( $param[0], $param[1] ... $param[n] );
	 * </code>
	 * <br />
	 * <br />
	 * $param of this method argument translates to callback.
	 * Type of $param is same as second argument of
	 * call_user_func_array function.<br />
	 * Its return value is returned as return value of this method.
	 * If callback in the instance can't call,
	 * BadMethodCallException is thrown.
	 *
	 * @param array $param Callback parameter
	 * @return mixed Return value from callback
	 */
	public function exec( $param = array() )
	{
		$sts = $this->get();
		if( $sts == self::NONE ){
			throw new BadMethodCallException;
		}
		elseif( $sts == self::ILLEGAL ){
			throw new RuntimeException;
		}

		return call_user_func_array( $this->_call, $param );
	}
}
?>
