<?php
/**
 * Event class
 *
 * This file defines YggEvent class.
 * It includes methods to manage event.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );


/**
 * Event class
 *
 * This class has methods to manage event.
 * This object can register and call any callbacks.
 *
 * <code>
 * $evtobj = new YggEvent;
 * $evtobj->add( array( &$handlerobj, "eventClick" ) );
 * $evtobj->add( "eventClick" );
 * $evtobj->exec( array( "test", &$refdata ) );</pre>
 * </code>
 *
 * @package YggDore.Base
 */
class YggEvent implements IteratorAggregate {
	/**
	 * Event callbacks
	 */
	private $_call;

	
	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->clear();
	}


	/**
	 * Get iterator
	 *
	 * This method returns iterator which has all callbacks in the instance.
	 * ArrayIterator has pair set whose key is event ID and value is callback.
	 *
	 * @return ArrayIterator
	 */
	public function getIterator()
	{
		return new ArrayIterator( $this->_call );
	}


	/**
	 * Add a callback
	 *
	 * A callback is added to the instance.<br />
	 * The argument can be specified type of callback which
	 * is function or object method.<br />
	 * It is same as argument of is_callable and call_user_func_array function.<br />
	 * The event ID is returns and it uses at YggEvent::delete method.
	 * This method throws UnexpectedValueException or RuntimeException when error occurs.
	 *
	 * @param array $callback Callback
	 * @return string Event ID
	 */
	public function add( $callback )
	{
		if( !is_callable( $callback ) ){
			throw new UnexpectedValueException;
		}
		
		for( $i = 0; $i < 100; $i++ ){
			$eh = YggTool::uniqueID();
			if( !isset($this->_call[$eh]) ){
				break;
			} 
		} 
		if( $i >= 100 ){
			throw new RuntimeException;
		}
		
		$this->_call[$eh] = $callback;
		
		return $eh;
	}


	/**
	 * Delete a callback
     *
	 * This method deletes a callback in the instance whose ID is $eh.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param string eh Event ID
	 */
	public function delete( $eh )
	{
		if( $eh == "" ){
			throw new UnexpectedValueException;
		}
		elseif( !isset($this->_call[$eh]) ){
			throw new UnexpectedValueException;
		}

		unset( $this->_call[$eh] );
	}
	

	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_call = array();
	}


	/**
	 * Execute callbacks
     *
	 * Execute all event callbacks.<br />
	 * The callback is called as follows.<br />
	 * <br />
	 * <code>
	 * callback( $param[0], $param[1] ... $param[n] );
	 * </code>
	 * <br />
	 * <br />
	 * $param of this method argument translates to callback.
	 * Type of $param is same as second argument of
	 * call_user_func_array function.<br />
	 * Its return value is set to temporary array and
	 * it returns to you when all of callbacks processing is completed.
	 * It is array that has pair list of event ID and return value.<br />
	 * If calling callbacks is failure, this method is thrown RuntimeException.
	 * 
	 * @param array $param Callback parameter
	 * @return array Array that has pair list of event ID and return value.
	 */
	public function exec( $param = null )
	{
		if( count( $param ) <= 0 ){
			$param = array();
		}

		foreach( $this->_call as $crec ){
			if( !is_callable($crec) ){
				throw new RuntimeException;
			} 
		}

		$rts = array();
		foreach( $this->_call as $key => $crec ){
			$rts[$key] = call_user_func_array( $crec, $param );
		}

		return $rts;
	}
}
?>
