<?php
/**
 * Image class used GD
 *
 * This file defines YggGDImage class.
 * It includes methods to create and operate png and gif, jpeg file and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggInt class
 */
require_once( "YggDore/Base/YggInt.php" );


/**
 * Image class used GD
 *
 * This class has methods to create and operate png and gif, jpeg file and so on.
 *
 * <code>
 * $gdiobj = YggGDImage::open( "/src/home.png" );
 * $gdiobj->saveJPEG( "/dst/dst.jpg" );
 * $gdiobj->close();
 * </code>
 *
 * @package YggDore.Base
 */
class YggGDImage {
	/**
	 * Resize mode : width base
	 *
	 * This value is used by YggGDImage::openRelResize method.
	 * It shows that resizing caluculation is width base.
	 */
	const WIDTH = 1;
	/**
	 * Resize mode : height base
	 *
	 * This value is used by YggGDImage::openRelResize method.
	 * It shows that resizing caluculation is height base.
	 */
	const HEIGHT = 2;
	/**
	 * Resize mode : shorter side base
	 *
	 * This value is used by YggGDImage::openRelResize method.
	 * It shows that resizing caluculation is used shorter side base.
	 */
	const SHORT = 3;
	/**
	 * Resize mode : longer base
	 *
	 * This value is used by YggGDImage::openRelResize method.
	 * It shows that resizing caluculation is used longer side base.
	 */
	const LONG = 4;

	/**
	 * Changing condition : less
	 *
	 * This value is used by YggGDImage::openRelResize method.
	 * It shows that size of target picture is less than argument size.
	 */
	const LT = 1;
	/**
	 * Changing condition : greater
	 *
	 * This value is used by YggGDImage::openRelResize method.
	 * It shows that size of target picture is greater than argument size.
	 */
	const GT = 2;
	/**
	 * Changing condition : equal
	 *
	 * This value uses YggGDImage::openRelResize method.
	 * It shows that size of target picture is argument size.
	 */
	const EQ = 3;


	/**
	 * Image ID
	 */
	private $_iid;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->_iid = null;
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_iid === null ){
			return false;
		}

		return true;
	}


	/**
	 * Get image width
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException or RuntimeException when error occurs.
	 *
	 * @return integer
	 */
	public function getWidth()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$width = imagesx( $this->_iid );
		if( !is_int( $width ) ){
			throw new RuntimeException;
		}

		return $width;
	}


	/**
	 * Get image height
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException or RuntimeException when error occurs.
	 *
	 * @return integer
	 */
	public function getHeight()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$height = imagesy( $this->_iid );
		if( !is_int( $height ) ){
			throw new RuntimeException;
		}

		return $height;
	}


	/**
	 * Open image
	 *
	 * This method creates image area whose
	 * pixel size is specified by $width and $height to memory.<br />
	 * If $istc is specified TRUE, color mode becomes TRUE color mode.<br />
	 * YggGDImage::close method must be called after this method calling.
	 *
	 * @param integer $width Width
	 * @param integer $height Height
	 * @param boolean $istc True color image mode
	 * @return YggGDImage Image object
	 * @see YggGDImage::close
	 */
	public static function open( $width, $height, $istc = true )
	{
		if( $width < 1 ){
			throw new RangeException;
		}
		elseif( $height < 1 ){
			throw new RangeException;
		}

		if( $istc ){
			if( !is_callable( "imagecreatetruecolor" ) ){
				throw new BadMethodCallException;
			}
			$iid = imagecreatetruecolor( $width, $height );
		}
		else{
			$iid = imagecreate( $width, $height );
		}
		if( !$iid ){
			throw new RuntimeException;
		}

		$obj = new YggGDImage;
		$obj->_iid = $iid;

		return $obj;
	}


	/**
	 * Close resource of image
	 *
	 * This method releases image resource in the instance.<br />
	 * This method throws RuntimeException or BadMethodCallException
	 * when error occurs.
	 *
	 * @see YggGDImage::open
	 */
	public function close()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$rts = imagedestroy( $this->_iid );
		if( !$rts ){
			throw new RuntimeException;
		}

		$this->_iid = null;
	}


	/**
	 * Open a image copyed and resized target image
	 *
	 * This method creates image copy of $bobj and
	 * the picture is resized to $width and $height.<br />
	 * If $istc is specified TRUE, color mode becomes TRUE color mode.<br />
	 * If $isrs is specified TRUE, pixel color is used resampling pixel color.(very heavy)<br />
	 * YggGDImage::close method must be called after this method calling.
	 *
	 * @param YggGDImage $bobj Source image
	 * @param integer $width Resize width
	 * @param integer $height Resize height
	 * @param boolean $istc True color image mode
	 * @param boolean $isrs Resampling mode
	 * @return YggGDImage Image object
	 * @see YggGDImage::close
	 */
	public static function openResize( $bobj, $width, $height, $istc = true, $isrs = true )
	{
		if( !$bobj->isSetIns() ){
			throw new BadMethodCallException;
		}

		$obj = self::open( $width, $height, $istc );

		try{
			$obj->copyResize( $bobj, 0, 0, 0, 0, 0, 0, 0, 0, $isrs );
		}
		catch( Exception $e ){
			$obj->close();
			throw $e;
		}

		return $obj;
	}


	/**
	 * Create a image relatively resized 
	 *
	 * This method creates image copy of $bobj and the picture is resized.<br />
	 * The base (Either of width or height is decided by $ckbs) side is resized to $picsz,
	 * and other side is resized relativity.<br />
	 * $fitmode is specified resize mode.
	 * If $fitmode is specified YggGDImage::LT,
	 * side of the target picture is resized to less than argument size.
	 * If $fitmode is specified YggGDImage::GT,
	 * side of the target picture is resized to greater than argument size.
	 * If $fitmode is specified YggGDImage::EQ,
	 * side of the target picture is resized to argument size.<br />
	 * If $istc is specified TRUE, color mode becomes TRUE color mode.<br />
	 * If $isrs is specified TRUE, pixel color is used resampling pixel color.(very heavy)<br />
	 * YggGDImage::close method must be called after this method calling.
	 *
	 * @param YggGDImage $bobj Source image
	 * @param integer $picsz Base pixel size
	 * @param integer $ckbs Resizing base<br />
	 * YggGDImage::WIDTH Width base<br />
	 * YggGDImage::HEIGHT Height base<br />
	 * YggGDImage::SHORT Shorter side base<br />
	 * YggGDImage::LONG Longer side base
	 * @param integer $fitmode Changing condition mode<br />
	 * YggGDImage::LT Less resize mode<br />
	 * YggGDImage::GT Greater resize mode<br />
	 * YggGDImage::EQ Normal resize mode
	 * @param boolean $istc True color image mode
	 * @param boolean $isrs Resampling mode
	 * @return YggGDImage Image object
	 * @see YggGDImage::close
	 */
	public static function openRelResize(
		$bobj,
		$picsz,
		$ckbs = self::SHORT,
		$fitmode = self::EQ,
		$istc = true,
		$isrs = true
	)
	{
		if( !$bobj->isSetIns() ){
			throw new BadMethodCallException;
		}
		elseif( $picsz < 1 ){
			throw new RangeException;
		}
		elseif( $ckbs < self::WIDTH ||
		        $ckbs > self::LONG ){
			throw new RangeException;
		}
		elseif( $fitmode < self::LT ||
		        $fitmode > self::EQ ){
			throw new RangeException;
		}

		$bwidth = $bobj->getWidth();
		if( $bwidth < 1 ){
			throw new UnexpectedValueException;
		}
		$bheight = $bobj->getHeight();
		if( $bheight < 1 ){
			throw new UnexpectedValueException;
		}

		if( $ckbs == self::SHORT ){
			$ckbs = self::WIDTH;
			if( $bwidth > $bheight ){
				$ckbs = self::HEIGHT;
			}
		}
		elseif( $ckbs == self::LONG ){
			$ckbs = self::WIDTH;
			if( $bwidth < $bheight ){
				$ckbs = self::HEIGHT;
			}
		}

		$bspic  =& $bwidth;
		$obspic =& $bheight;
		if( $ckbs == self::HEIGHT ){
			$bspic  =& $bheight;
			$obspic =& $bwidth;
		}

		$isrcal = true;
		if( $fitmode == self::LT ){
			if( $bspic <= $picsz ){
				$isrcal = false;
			}
		}
		elseif( $fitmode == self::GT ){
			if( $bspic >= $picsz ){
				$isrcal = false;
			}
		}

		if( $isrcal ){
			// YggInt::extract is used to guard overflow
			$obspic = YggInt::extract( $obspic * ( $picsz / $bspic ) );
			if( $obspic <= 0 ){
				$obspic = 1;
			}
			$bspic = $picsz;
		} 

		return self::openResize(
			$bobj, $bwidth, $bheight, $istc, $isrs
		);
	}


	/**
	 * Open image file
	 *
	 * Open file specified by $ipath.<br />
	 * YggGDImage::close method must be called after this method calling.
	 *
	 * @param string $ipath Image file path
	 * @return YggGDImage Image object
	 * @see YggGDImage::close
	 */
	public static function openFile( $ipath )
	{
		if( $ipath == "" ){
			throw new UnexpectedValueException;
		}
		elseif( !is_file( $ipath ) ){
			throw new UnexpectedValueException;
		}

		$sifttbl = array(
			IMAGETYPE_GIF  => "imagecreatefromgif",
			IMAGETYPE_JPEG => "imagecreatefromjpeg",
			IMAGETYPE_PNG  => "imagecreatefrompng"
		);

		$imgi = self::analyFile( $ipath );
		if( !isset($sifttbl[$imgi[2]]) ){
			throw new UnexpectedValueException;
		}
		elseif( !is_callable($sifttbl[$imgi[2]]) ){
			throw new BadMethodCallException;
		}

		$iid = call_user_func( $sifttbl[$imgi[2]], $ipath );
		if( !$iid ){
			throw new RuntimeException;
		}
		
		$obj = new YggGDImage;
		$obj->_iid = $iid;

		return $obj;
	}


	/**
	 * Copy and resize image
	 *
	 * This method is copied $bobj to image resource in the instance.<br />
	 * Source point and size is specified by $bx and $by, $bwidth, $bheight.<br />
	 * Destination point and size is specified by $cx and $cy, $cwidth, $cheight.<br />
	 * If $isrs is specified TRUE, pixel color is used resampling pixel color.(very heavy)
	 *
	 * @param YggGDImage $bobj Source image
	 * @param integer $bx X-coordinate of source point
	 * @param integer $by Y-coordinate of source point
	 * @param integer $bwidth Source width(0 means width size)
	 * @param integer $bheight Source height(0 means height size)
	 * @param integer $cx X-coordinate of destination point
	 * @param integer $cy Y-coordinate of destination point
	 * @param integer $cwidth Destination width(0 means width size)
	 * @param integer $cheight Destination height(0 means height size)
	 * @param boolean $isrs Resampling mode
	 */
	public function copyResize(
		$bobj,
		$bx = 0, $by = 0, $bwidth = 0, $bheight = 0,
		$cx = 0, $cy = 0, $cwidth = 0, $cheight = 0,
		$isrs = true
	)
	{
		$chktbl = array(
			&$bx, &$by, &$bwidth, &$bheight,
			&$cx, &$cy, &$cwidth, &$cheight
		);

		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		if( !$bobj->isSetIns() ){
			throw new UnexpectedValueException;
		}
		for( $i = 0; $i < count($chktbl); $i++ ){
			if( $chktbl[$i] < 0 ){
				throw new RangeException;
			}
		}

		if( $bwidth == 0 ){
			$bwidth = $bobj->getWidth();
			if( $bwidth < 1 ){
				throw new UnexpectedValueException;
			}
		}
		if( $bheight == 0 ){
			$bheight = $bobj->getHeight();
			if( $bheight < 1 ){
				throw new UnexpectedValueException;
			}
		}
		if( $cwidth == 0 ){
			$cwidth = $this->getWidth();
			if( $cwidth < 1 ){
				throw new UnexpectedValueException;
			}
		}
		if( $cheight == 0 ){
			$cheight = $this->getHeight();
			if( $cheight < 1 ){
				throw new UnexpectedValueException;
			}
		}

		if( $isrs ){
			if( !is_callable( "imagecopyresampled" ) ){
				throw new BadMethodCallException;
			}
			$rts = imagecopyresampled(
				$this->_iid, $bobj->_iid,
				$cx, $cy, $bx, $by,
				$cwidth, $cheight, $bwidth, $bheight
			);
		}
		else{
			$rts = imagecopyresized(
				$this->_iid, $bobj->_iid,
				$cx, $cy, $bx, $by,
				$cwidth, $cheight, $bwidth, $bheight
			);
		}
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/**
	 * Save file whose type is gif
	 *
	 * Save file to $dpath. It is gif format.<br />
	 * The umask influences its permisson.
	 *
	 * @param string $dpath Save file path
	 */
	public function saveGIF( $dpath )
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		if( $dpath == "" ){
			throw new UnexpectedValueException;
		}

		$rts = imagegif( $this->_iid, $dpath );
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/**
	 * Save file whose type is jpeg
	 *
	 * Save file to $dpath. It is jpeg format.<br />
	 * $quality is specified ranges
	 * from 0 (worst quality, smaller file)
	 * to 100 (best quality, biggest file).<br />
	 * The umask influences its permisson.
	 *
	 * @param string $dpath Save file path
	 * @param string $quality Image quality
	 */
	public function saveJPEG( $dpath, $quality = 75 )
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		if( $dpath == "" ){
			throw new UnexpectedValueException;
		}
		elseif( $quality < 0 || $quality > 100 ){
			throw new RangeException;
		}

		$rts = imagejpeg( $this->_iid, $dpath, $quality );
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/**
	 * Save file whose type is png
	 *
	 * Save file to $dpath. It is png format.<br />
	 * The umask influences its permisson.
	 *
	 * @param string $dpath Save file path
	 */
	public function savePNG( $dpath )
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		if( $dpath == "" ){
			throw new UnexpectedValueException;
		}

		$rts = imagepng( $this->_iid, $dpath );
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/**
	 * Analize image file
	 *
	 * Image file specified by $ipath is analized.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param string $ipath Image file path
	 * @return array Same getimagesize() function return value
	 */
	public static function analyImage( $ipath )
	{
		if( $ipath == "" ){
			throw new UnexpectedValueException;
		}

		$analy = getimagesize( $ipath );
		if( !is_array( $analy ) ){
			throw new UnexpectedValueException;
		}

		return $analy;
	}
}
?>
