<?php
/**
 * HTML(XHTML) tag and URL creation class
 *
 * This file defines YggHTML class.
 * It includes methods to create HTML(XHTML) tag and URL and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * HTML(XHTML) tag and URL creation class
 *
 * This class has methods to create HTML(XHTML) tag and URL and so on.
 * Now, all methods in this class are static.
 *
 * @package YggDore.Base
 */
class YggHTML {
	/**
	 * Escape HTML special characters
	 *
	 * This method is same as "htmlspecialchars($str, ENT_QUOTES)".
	 *
	 * @param string $str Same as first argument of htmlspecialchars
	 * @return string
	 */
	public static function specialChars( $str )
	{
		return htmlspecialchars( $str, ENT_QUOTES );
	}


	/**
	 * Convert to HTML entities
	 *
	 * This method is same as "htmlentities($str, ENT_QUOTES, $cs, $de)".
	 *
	 * @param string $str Same as first argument of htmlentities
	 * @return string
	 */
	public static function entities( $str )
	{
		return htmlentities( $str, ENT_QUOTES );
	}


	/**
	 * Create HTML(XHTML) hidden tag
	 *
	 * This method creates HTML(XHTML) hidden tag from $data.<br />
	 * $exttbl is target key name list in $data.<br />
	 * Default setting is that name of tag is used key name of each $data elements.
	 * Hidden tag name can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * If $isx is specified TRUE, XHTML tag creates.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $exttbl Target key list
	 * @param array $kntbl Tag name list
	 * @param boolean $isx XHTML mode
	 * @return string
	 */
	public static function createHiddenTag( $data, $exttbl, $kntbl = null, $isx = true )
	{
		if( count( $exttbl ) <= 0 ){
			throw new UnexpectedValueException;
		}

		$xtag = "";
		if( $isx ){
			$xtag = " /";
		}

		$tag = "";
		foreach( $exttbl as $key ){
			if( !isset( $data[$key] ) ){
				continue;
			}
			$val = $data[$key];

			$name = $key;
			if( isset($kntbl[$key]) ){
				$name = $kntbl[$key];
			}
			if( $name == "" ){
				throw new UnexpectedValueException;
			}

			$name = self::specialChars( $name );
			$val  = self::specialChars( $val );

			$tag .= '<input type="hidden" ' .
			        'name="' . $name . '" ' .
			        'value="' . $val . '"' .
					$xtag . '>';
		}

		return $tag;
	}


	/**
	 * Create HTML(XHTML) hidden tags from all elements in array
	 *
	 * This method creates HTML(XHTML) hidden tag from all elements in $data.<br />
	 * Default setting is that name of tag is used key name of each $data elements.
	 * Hidden tag name can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * If $isx is specified TRUE, XHTML tag creates.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $kntbl Tag name list
	 * @param boolean $isx XHTML mode
	 * @return string
	 */
	public static function createAllHiddenTag( $data, $kntbl = null, $isx = true )
	{
		if( count( $data ) <= 0 ){
			return "";
		}

		return self::createHiddenTag(
			$data, array_keys( $data ), $kntbl, $isx
		);
	}


	/**
	 * Create URL encode
	 *
	 * This method creates URL encode from $data.
	 * $exttbl is target key name list in $data.<br />
	 * Default setting is that key name of URL encode is used key name of each $data elements.
	 * Key name of URL encode can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $exttbl Target key list
	 * @param array $kntbl Key name list
	 * @return string
	 */
	public static function createURLEncode( $data, $exttbl, $kntbl = null )
	{
		if( count( $exttbl ) <= 0 ){
			throw new UnexpectedValueException;
		}

		$rts = "";
		$asp = "";
		foreach( $exttbl as $key ){
			if( !isset( $data[$key] ) ){
				continue;
			}
			$val = $data[$key];

			$name = $key;
			if( isset($kntbl[$key]) ){
				$name = $kntbl[$key];
			}
			if( $name == "" ){
				throw new UnexpectedValueException;
			}

			$name = urlencode( $name );
			$val  = urlencode( $val );

			$rts .= $asp . $name . "=" . $val;

			$asp = "&";
		}

		return $rts;
	}


	/**
	 * Create URL encode from all elements in array
	 *
	 * This method creates URL encode from all elements in $data.
	 * $exttbl is target key name list in $data.<br />
	 * Default setting is that key name of URL encode is used key name of each $data elements.
	 * Key name of URL encode can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $kntbl Key name list
	 * @return string
	 */
	public static function createAllURLEncode( $data, $kntbl = null )
	{
		if( count( $data ) <= 0 ){
			return "";
		}

		return self::createURLEncode(
			$data, array_keys( $data ), $kntbl
		);
	}
}
?>
