<?php
/**
 * HTTP client class
 *
 * This file defines YggHTTPClient class.
 * It includes class which speaks HTTP and connect remote host, recieve data.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggSafe class
 */
require_once( "YggDore/Base/YggSafe.php" );



/**
 * HTTP client class
 *
 * This class speaks HTTP and connect remote host, recieve data.
 *
 * <code>
 * $hc = YggHTTPClient::open(
 *     "http://www.yggdore.com/test.cgi",
 *     array( "hogehoge" => "testdata" ),
 *     true
 * );
 * $hc->get( 8192 );
 * $hc->close();
 * </code>
 *
 * @package YggDore.Base
 */
class YggHTTPClient {
	/**
	 * Connection resource
	 */
	private $_conn;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->_conn = null;
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_conn === null ){
			return false;
		}

		return true;
	}


	/**
	 * Open HTTP client
	 * 
	 * This method parses URL and connect the host.<br />
	 * $url is specified destination URL.<br />
	 * $data is query array data. These data is converted to URL encode.<br />
	 * If $isp is specified TRUE, this class sends POST query format,
	 * otherwise send GET query format.
	 *
	 * @param string $url Destination URL
	 * @param array $data Query data<br />
	 * (Its each records has the following elements)<br />
	 * key : (string) Query key name<br />
	 * value : (string) Query value
	 * @param boolean $isp POST query mode
	 * @return YggHTTPClient
	 */
	public static function open( $url, $data = null, $isp = false )
	{
		$sndh = "GET";
		if( $isp ){
			$sndh = "POST";
			if( count($data) <= 0 ){
				throw new UnexpectedValueException;
			}
		}

		$purl = parse_url( $url );
		if( !is_array( $purl ) ){
			throw new UnexpectedValueException;
		}

		$cet = array( "scheme", "host" );
		foreach( $cet as $key ){
			if( !isset( $purl[$key] ) ||
				!preg_match( '/^[[:graph:]]+$/', $purl[$key] ) ){
				throw new UnexpectedValueException;
			}
		}

		if( !isset( $purl['path'] ) ||
			!preg_match( '/^[[:graph:]]+$/', $purl['path'] ) ){
			$purl = "/";
		}

		$purl['scheme'] = strtolower($purl['scheme']);
		if( $purl['scheme'] != "http" && $purl['scheme'] != "https" ){
			throw new UnexpectedValueException;
		}

		if( !isset( $purl['port'] ) ){
			$purl['port'] = 80;
			if( $purl['scheme'] == "https" ){
				$purl['port'] = 443;
			}
		}
		else{
			$purl['port'] = YggSafe::safeInt( $purl['port'] );
		}
		
		$sslmd = "";
		if( $purl['scheme'] == "https" ){
			$sslmd = "ssl://";
		}

		$query = "";
		if( count($data) > 0 ){
			$query = self::createQuery( $data );

			// If query type is GET, create URL from path and data
			if( !$isp ){
				$purl['path'] .= "?" . $query;
				$query         = "";
			}
		}

		$conn = fsockopen(
			$sslmd . $purl['host'],
			$purl['port'],
			$errno,
			$errstr,
			90
		);
		if( !$conn ){
			throw new RuntimeException;
		}

		try{
			$rts = stream_set_blocking( $conn, true );
			if( !$rts ){
				throw new RuntimeException;
			}

			$rts = stream_set_timeout( $conn, 90 );
			if( !$rts ){
				throw new RuntimeException;
			}

			$sndh .= " " . $purl['path'] . " HTTP/1.0\r\n" .
			         "HOST: " . $purl['host'] . "\r\n" .
			         "User-Agent: YggHTTPClient-0.0.1\r\n";
			if( $isp ){
				$sndh .= "Content-Type: " .
				         "application/x-www-form-urlencoded\r\n" .
						 "Content-Length: " . strlen( $query ) . "\r\n";
			}
			$sndh .= "\r\n";
			$rts = fwrite( $conn, $sndh );
			if( !$rts ){
				throw new RuntimeException;
			}

			if( $data != "" ){
				$rts = fwrite( $conn, $query );
				if( !$rts ){
					throw new RuntimeException;
				}
			}

			$line = stream_get_line( $conn, 8192, "\r\n" );
			if( $line === false ){
				throw new RuntimeException;
			}
			$info = stream_get_meta_data( $conn );
			if( !$info || $info['timed_out'] ){
				throw new RuntimeException;
			}
			elseif( !preg_match( '/^HTTP\/[0-9\.]+[\s]+200/', $line ) ){
				throw new RuntimeException;
			}

			while( !feof( $conn ) ){
				$line = stream_get_line( $conn, 8192, "\r\n" );
				if( $line === false ){
					throw new RuntimeException;
				}

				$info = stream_get_meta_data( $conn );
				if( !$info || $info['timed_out'] ){
					throw new RuntimeException;
				}

				if( $line == "" ){
					break;
				}
			}
		}
		catch( Exception $e ){
			fclose( $conn );
			throw $e;
		}

		$hc        = new YggHTTPClient;
		$hc->_conn = $conn;

		return $hc;
	}


	/**
	 * Close HTTP connection
	 */
	public function close()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$rts = fclose( $this->_conn );
		if( !$rts ){
			throw new RuntimeException;
		}

		$this->_conn = null;
	}


	/**
	 * Is EOF
	 *
	 * This method checks whether stream cursor has encounterd at the end of file.
	 * 
	 * @return boolean
	 */
	public function isEOF()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return feof( $this->_conn );
	}


	/**
	 * Get result data
	 * 
	 * @param integer $size Max response data size ( -1 means all of response data )
	 * @return string Response data
	 */
	public function get( $size )
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		if( $size < -1 || $size == 0 ){
			throw new UnexpectedValueException;
		}

		try{
			$rdata = stream_get_contents( $this->_conn, $size );
			if( $rdata === false ){
				throw new RuntimeException;
			}

			$info = stream_get_meta_data( $this->_conn );
			if( !$info || $info['timed_out'] ){
				throw new RuntimeException;
			}
		}
		catch( Exception $e ){
			fclose( $this->_conn );
			throw $e;
		}

		return $rdata;
	}


	/**
	 * Create query
	 * 
	 * This method converts array data to urlencoding.
	 *
	 * @param array $data Query data<br />
	 * (Its each records has the following elements)<br />
	 * key : (string) Query key name<br />
	 * value : (string) Query value
	 * @return string Query string
	 */
	public static function createQuery( $data )
	{
		$rts = "";
		$sp  = "";
		foreach( $data as $datar ){
			$rts .= $sp . urlencode( $datar['key'] ) . "=" . urlencode( $datar['value'] );
			$sp   = "&";
		}

		return $rts;
	}
}
?>
