<?php
/**
 * Integer value operation class
 *
 * This file defines YggInt class.
 * It includes methods to check integer value and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggNumeric class
 */
require_once( "YggDore/Base/YggNumeric.php" );


/**
 * Integer value operation class
 *
 * This class has methods to check integer value and so on.
 *
 * @package YggDore.Base
 */
class YggInt {
	/**
	 * Convert from digit string that has comma to integer value
	 *
	 * This method converts from digit string of $data that has comma to integer value.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $data = "-1,050";
	 * $rts  = YggInt::conv( $data );
	 * (Result)
	 * $rts => -1050
	 * </code>
	 *
	 * @param string $data Target string
	 * @return integer Result integer value
	 */
	public static function conv( $data )
	{
		$data = YggNumeric::conv( $data );
		if( !is_int( $data ) ){
			throw new UnexpectedValueException;
		}

		return $data;
	}


	/**
	 * Check data is integer value
	 *
	 * This method checks whether value of $data is integer value.<br />
	 * This method returns TRUE if $data is type of integer, otherwise FALSE.<br />
	 * Range of integer is defined YGG_INTMAX and YGG_INTMIN in Config.php.<br />
	 * If type of $data is float value, returns FALSE.
	 * But YggInt::isRange method returns TRUE.
	 *
	 * <code>
	 * $data = 1234.5;
	 * $rts  = YggInt::is( $data );
	 * (Result)
	 * $rts => false</pre>
	 * </code>
	 * 
	 * @param numeric $data Target value
	 * @return boolean If $data is integer type, returns TRUE, otherwise return FALSE
	 * @see YggInt::isRange()
	 */
	public static function is( $data )
	{
		if( $data < YGG_INTMIN || $data > YGG_INTMAX ){
			return false;
		}
		elseif( (int)$data != (float)$data ){
			return false;
		}

		return true;
	}


	/**
	 * Check range of integer
	 *
	 * This method checks whether value of $data is range of integer.<br />
	 * This method returns TRUE if $data is range of integer, otherwise FALSE.<br />
	 * Range of integer is defined YGG_INTMAX and YGG_INTMIN in Config.php.<br />
	 * If type of $data is float value, returns TRUE.
	 * But YggInt::is method returns TRUE.
	 *
	 * <code>
	 * $data = 1234.5;
	 * $rts  = YggInt::isRange( $data );
	 * (Result)
	 * $rts => true
	 * </code>
	 *
	 * @param numeric $data Target value
	 * @return boolean Return TRUE if $data is range of integer, otherwise return FALSE
	 * @see YggInt::is()
	 */
	public static function isRange( $data )
	{
		if( $data < YGG_INTMIN || $data > YGG_INTMAX ){
			return false;
		}

		return true;
	}


	/**
	 * Extract part of integer
	 *
	 * Extract part of integer.<br />
	 * Range of integer is defined YGG_INTMAX and YGG_INTMIN in Config.php.<br />
	 * This method throws RangeException when $data isn't range of integer.
	 *
	 * <code>
	 * $data = 1234.5;
	 * $rts  = YggInt::extract( $data );
	 * (Result)
	 * $rts => 1234
	 * </code>
	 * 
	 * @param numeric $data Target value(integer or float type)
	 * @return integer Result integer value
	 */
	public static function extract( $data )
	{
		if( !self::isRange( $data ) ){
			throw new RangeException;
		}

		return (int)$data;
	}
}
?>
