<?php
/**
 * List class
 *
 * This file defines YggList class.
 * It includes methods to manage array data.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggDore configuration file
 */
require_once( "YggDore/Base/Config.php" );



/**
 * List class
 *
 * This class has methods to manage array data.
 *
 * <code>
 * $dataobj = new YggList;
 * $dataobj->setSelect(
 *     array(
 *         "lchk_1" => 1,
 *         "lmuk_1" => 15,
 *         "lchk_2" => 0,
 *         "lmuk_2" => "aiueo"
 *     ),
 *     2,
 *     "lchk_", "",
 *     "lmuk_", "",
 * );
 * $data = $dataobj->get();
 * </code>
 *
 * @package YggDore.Base
 */
class YggList implements IteratorAggregate {
	/**
	 * List data
	 */
	private $_list;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct() 
	{
		$this->clear();
	}


	/**
	 * Get iterator
	 *
	 * This method returns iterator that has all list elements.
	 *
	 * @return ArrayIterator
	 */
	public function getIterator()
	{
		return new ArrayIterator( $this->_list );
	}


	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_list = null;
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_list === null ){
			return false;
		}

		return true;
	}


	/**
	 * Set value
	 *
	 * This method sets $list to the instance.<br />
	 * $list has some elements that is type of integer.
	 *
	 * @param array $list List
	 */
	public function set( $list )
	{
		$this->_list = $list;
	}


	/**
	 * Set value selected
	 *
	 * This method searches selected value and sets them to the instance.<br />
	 * $data is array that has selection part and value part data.<br />
	 * The Selection part is $data[($selkh)(n)($selkf)].<br />
	 * The Value part is $data[($valkh)(n)($valkf)].<br />
	 * "n" increments from 1 to $datac.<br />
	 * When value of the selection part is TRUE,
	 * this method extracts value of the value part and sets these value to the instance.
	 *
	 * @param array $data Target data
	 * @param integer $datac Number of elements in $list
	 * @param string selkh Header of key name of selection part
	 * @param string selkf Footer of key name of selection part
	 * @param string valkh Header of key name of value part
	 * @param string valkf Footer of key name of value part
	 */
	public function setSelect( $data, $datac, $selkh = "", $selkf = "", $valkh = "", $valkf = "" )
	{
		if( $datac <= 0 ){
			return false;
		}

		$list = array();
		for( $i = 1; $i <= $datac; $i++ ){
			if( !$data[$selkh . $i . $selkf] ){
				continue;
			}

			$list[] = $data[$valkh . $i . $valkf];
		}

		$this->set( $list );

		return true;
	}


	/**
	 * Get list data
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array
	 */
	public function get()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_list;
	}


	/**
	 * Get number of elements in the list
	 *
	 * This method returns number of elements in the list of instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array
	 */
	public function getCount()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return count( $this->_list );
	}
}
?>
