<?php
/**
 * Integer list class
 *
 * This file defines YggListInt class.
 * It includes methods to manage integer values.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggDore configuration file
 */
require_once( "YggDore/Base/Config.php" );
/**
 * Require YggSafe class
 */
require_once( "YggDore/Base/YggSafe.php" );



/**
 * Integer list class
 *
 * This class has methods to manage integer values.
 *
 * <code>
 * $dataobj = new YggListInt;
 * $dataobj->setSelect(
 *     array(
 *         "lchk_1" => 1,
 *         "lmuk_1" => 15,
 *         "lchk_2" => 0,
 *         "lmuk_2" => 5
 *     ),
 *     2,
 *     "lchk_", "",
 *     "lmuk_", "",
 * );
 * $data = $dataobj->get();
 * </code>
 *
 * @package YggDore.Base
 */
class YggListInt implements IteratorAggregate {
	/**
	 * List data
	 */
	private $_list;
	/**
	 * Min value in the list
	 */
	private $_rmin;
	/**
	 * Max value in the list
	 */
	private $_rmax;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct() 
	{
		$this->clear();
	}


	/**
	 * Get iterator
	 *
	 * This method returns iterator that has all list elements.
	 *
	 * @return ArrayIterator
	 */
	public function getIterator()
	{
		return new ArrayIterator( $this->_list );
	}


	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_list = null;
		$this->_rmin = null;
		$this->_rmax = null;
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_list === null ){
			return false;
		}

		return true;
	}


	/**
	 * Set value
	 *
	 * This method sets $list to the instance.<br />
	 * $list has some elements that is type of integer.
	 *
	 * @param array $list List
	 */
	public function set( $list )
	{
		$rmin = YGG_INTMAX;
		$rmax = YGG_INTMIN;
		foreach( $list as $rec ){
			$rmin = min( $rmin, $rec );
			$rmax = max( $rmax, $rec );
		}

		$this->_list = $list;
		$this->_rmin = null;
		$this->_rmax = null;
		if( count($this->_list) > 0 ){
			$this->_rmin = $rmin;
			$this->_rmax = $rmax;
		}
	}


	/**
	 * Set value that has string elements
	 *
	 * This method converts $data from string type to integer type and
	 * sets $list to the instance.<br />
	 * $list has some elements that is type of integer string.
	 *
	 * @param array $list List set string type values
	 */
	public function setString( $data )
	{
		$list = array();
		foreach( $data as $rec ){
			$w = YggSafe::safeInt( $rec );
			$list[] = $w;
		}

		$this->set( $list );
	}


	/**
	 * Set value selected
	 *
	 * This method searches selected value and sets them to the instance.<br />
	 * $data is array that has selection part and value part data.<br />
	 * The Selection part is $data[($selkh)(n)($selkf)].<br />
	 * The Value part is $data[($valkh)(n)($valkf)].<br />
	 * "n" increments from 1 to $datac.<br />
	 * When value of the selection part is TRUE,
	 * this method extracts value of the value part and sets these value to the instance.
	 *
	 * @param array $data Target data
	 * @param integer $datac Number of elements in $list
	 * @param string selkh Header of key name of selection part
	 * @param string selkf Footer of key name of selection part
	 * @param string valkh Header of key name of value part
	 * @param string valkf Footer of key name of value part
	 */
	public function setSelect( $data, $datac, $selkh = "", $selkf = "", $valkh = "", $valkf = "" )
	{
		if( $datac <= 0 ){
			return false;
		}

		$list = array();
		for( $i = 1; $i <= $datac; $i++ ){
			if( !$data[$selkh . $i . $selkf] ){
				continue;
			}

			$list[] = $data[$valkh . $i . $valkf];
		}

		$this->set( $list );

		return true;
	}


	/**
	 * Set value selected that has string format elements
	 *
	 * This method searches selected value and sets them to the instance.<br />
	 * $data is array that has selection part and value part data.<br />
	 * The Selection part is $data[($selkh)(n)($selkf)].<br />
	 * The Value part is $data[($valkh)(n)($valkf)].<br />
	 * "n" increments from 1 to $datac. <br />
	 * When value of the selection part is TRUE,
	 * this method extracts and converts value of the value part to integer type and
	 * sets these value to the instance.
	 *
	 * @param array $data Target list data
	 * @param integer $datac Number of elements in $list
	 * @param string selkh Selection key name header
	 * @param string selkf Selection key name footer
	 * @param string valkh Value key name header
	 * @param string valkf Value key name footer
	 */
	public function setSelectString(
		$data, $datac,
		$selkh = "", $selkf = "",
		$valkh = "", $valkf = ""
	)
	{
		if( $datac <= 0 ){
			return false;
		}

		$list = array();
		for( $i = 1; $i <= $datac; $i++ ){
			if( !$data[$selkh . $i . $selkf] ){
				continue;
			}

			$list[] = $data[$valkh . $i . $valkf];
		}

		return $this->setString( $list );
	}


	/**
	 * Get list data
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array
	 */
	public function get()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_list;
	}


	/**
	 * Get number of elements in the list
	 *
	 * This method returns number of elements in the list of instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array
	 */
	public function getCount()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return count( $this->_list );
	}


	/**
	 * Get value range in the list
	 *
	 * This method returns range of the list values of instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array<br />
	 * (The array has the following elements)<br />
	 * max : Max value in the list<br />
	 * min : Min value in the list
	 */
	public function getRange()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return array(
			"min" => $this->_rmin,
			"max" => $this->_rmax
		);
	}


	/**
	 * Get min value in the list
	 *
	 * This method returns minimum value in the list of instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer
	 */
	public function getRangeMin()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_rmin;
	}


	/**
	 * Get max value in the list
	 *
	 * This method returns maximum value in the list of instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer
	 */
	public function getRangeMax()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_rmax;
	}
}
?>
