<?php
/**
 * Login and logout framework class
 *
 * This file defines YggLogio class.
 * It includes methods to control login and logout.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggRawWebRequest class
 */
require_once( "YggDore/Base/YggRawWebRequest.php" );
/**
 * Require YggDelegate class
 */
require_once( "YggDore/Base/YggDelegate.php" );
/**
 * Require YggEvent class
 */
require_once( "YggDore/Base/YggEvent.php" );
/**
 * Require YggSession class
 */
require_once( "YggDore/Base/YggSession.php" );
/**
 * Require YggLogioException class
 */
require_once( "YggDore/Base/YggLogioException.php" );


/**
 * Login and logout framework class
 *
 * This class has methods to control login and logout.<br />
 * This class has the following events and delegates.<br />
 *
 * <code>
 * void evtLogin( array $req, array $kpr, mixed $param, Exception $e );
 * </code>
 * This event is called when YggLogioException throws from delegate.<br />
 * $req is web request that has elements of $rkl of YggLogio::start method.<br />
 * $kpr is web request data exceeded elements of $mdk and $rkl of YggLogio::start method.<br />
 * $param is same as YggLogio::start argument.<br />
 * $e is exception.
 *
 * <code>
 * void evtError( Exception $e );
 * </code>
 * This event is called when error occurs or exception throws from delegate.
 *
 * <code>
 * string dgStable( string $val, mixed $param );
 * </code>
 * This delegate is called when web request data is optimized.
 * The return value is stable string.
 *
 * <code>
 * mixed dgLogin( array $req, mixed $param );
 * </code>
 * This delegate is called when login from web request is needed.
 * 
 * <code>
 * mixed dgSLogin( array $req, mixed $param );
 * </code>
 * This delegate is called when login from session is needed.
 * 
 * <code>
 * mixed dgLogout( array $req, mixed $param );
 * </code>
 * This delegate is called when logout is needed.
 * 
 * <code>
 * $mysqli = new YggMySQLi( "127.0.0.1", "user", "pass", "dbname" );
 * $authobj = new YggSessionMySQLi;
 * $authobj->start( "yggmail", $mysqli, "table" );
 * $lofobj = new YggLogio;
 * $lofobj->evtLogin->add( "loginview" );
 * $lofobj->dgStable->set( "logincallback" );
 * $lofobj->dgLogin->set( "logincallback" );
 * $lofobj->dgIsLogin->set( "logincallback" );
 * $lofobj->dgSLogin->set( "slogincallback" );
 * $lofobj->dgLogout->set( "logoutcallback" );
 * $lofobj->start(
 *     "mode",
 *     array( "id", "pass" )
 * );
 * </code>
 *
 * @package YggDore.Base
 */
class YggLogio {
	/**
	 * Login mode
	 */
	const LOGIN   = "1";

	/**
	 * Delegate for string stable
	 */
	public $dgStable;
	/**
	 * Delegate for login from web request data
	 */
	public $dgLogin;
	/**
	 * Delegate for login from session
	 */
	public $dgSLogin;
	/**
	 * Delegate for logout
	 */
	public $dgLogout;
	/**
	 * Event for login
	 */
	public $evtLogin;
	/**
	 * Event for error
	 */
	public $evtError;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->dgStable = new YggDelegate;
		$this->dgLogin  = new YggDelegate;
		$this->dgSLogin = new YggDelegate;
		$this->dgLogout = new YggDelegate;
		$this->evtLogin = new YggEvent;
		$this->evtError = new YggEvent;
	}


	/**
	 * Login
	 *
	 * This method starts to login.<br />
	 * Each delegate and event is called if necessary.<br />
	 * When $_REQUEST[$mdk] is YggLogio::LOGIN, YggLogio::dgLogin delegate is called,
	 * otherwise YggLogio::dgSLogin delegate is called.<br />
	 * When exception is thrown from each delegates, this method calls the following event.<br />
	 * YggLogioException => YggLogio::evtLogin.<br />
	 * Other exception => YggLogio::evtError.<br />
	 * $rkl is specified key name list. It is used to extract data from web request data.
	 * The data transrates to each delegates and events as $req.<br />
	 * Other web request data is set to $kpr. $kpc is maximum number of elements in $kpr.
	 *
	 * @param arraykey $mdk Mode key
	 * @param array $rkl Web request key
	 * @param mixed $param Deliver data to each event and delegate
	 * @param integer $kpc Keep data count
	 * @return mixed Return value of a delegate
	 */
	public function login( $mdk, $rkl, $param = null, $kpc = 256 )
	{
		$md  = null;
		$req = array();
		try{
			$wr =& YggRawWebRequest::getRequest();
			if( count($rkl) > 0 ){
				$req = YggArray::extract( $wr, $rkl );
			}
			$mda = YggArray::extract( $wr, array($mdk) );
			$md  = $mda[$mdk];

			$md = $this->dgStable->exec( array( $md, $param ) );
			foreach( array_keys($req) as $key ){
				$req[$key] = $this->dgStable->exec( array( $req[$key], $param ) );
			}
		}
		catch( Exception $e ){
			$this->evtError->exec( array( $e ) );
			throw $e;
		}

		$rparam = null;
		try{
			if( $md == self::LOGIN ){
				$rparam = $this->dgLogin->exec( array( $req, $param ) );
				return $rparam;
			}

			$rparam = $this->dgSLogin->exec( array( $req, $param ) );
		}
		catch( YggLogioException $e ){
			$rkl[] = $mdk;
			$kpr   = $this->_getKR( $kpc, $rkl, $param );

			$this->evtLogin->exec( array( $req, $kpr, $param, $e ) );
			throw $e;
		}
		catch( Exception $e ){
			$this->evtError->exec( array( $e ) );
			throw $e;
		}

		return $rparam;
	}


	/**
	 * Logout
	 *
	 * This method starts to logout.<br />
	 * YggLogio::dgSLogin delegate is called in this method.<br />
	 * When exception is thrown from each delegates, this method calls the following event.<br />
	 * YggLogioException => YggLogio::evtLogin.<br />
	 * Other exception => YggLogio::evtError.<br />
	 * $rkl is specified key name list. It is used to extract data from web request data.
	 * The data transrates to each delegates and events as $req.<br />
	 * Other web request data is set to $kpr. $kpc is maximum number of elements in $kpr.
	 *
	 * @param array $rkl Web request key
	 * @param mixed $param Deliver data to each event and delegate callback
	 * @param integer $kpc Keep data count
	 * @return mixed Return value of a delegate
	 */
	public function logout( $rkl, $param = null, $kpc = 256 )
	{
		$req = array();
		try{
			$wr =& YggRawWebRequest::getRequest();
			if( count($rkl) > 0 ){
				$req = YggArray::extract( $wr, $rkl );
			}

			foreach( array_keys($req) as $key ){
				$req[$key] = $this->dgStable->exec( array( $req[$key], $param ) );
			}
		}
		catch( Exception $e ){
			$this->evtError->exec( array( $e ) );
			throw $e;
		}

		$rparam = null;
		try{
			$rparam = $this->dgLogout->exec( array( $req, $param ) );
		}
		catch( YggLogioException $e ){
			$kpr = $this->_getKR( $kpc, $rkl, $param );

			$this->evtLogin->exec( array( $req, $kpr, $param, $e ) );
			throw $e;
		}
		catch( Exception $e ){
			$this->evtError->exec( array( $e ) );
			throw $e;
		}

		return $rparam;
	}


	/**
	 * Get web keep request data
	 *
	 * This method creates and returns web request data excluded $rkl.
	 * $kpc is maximum number of elements in $kpr.
	 * This data can be used as $kpr of YggLogio::start delegate.
	 *
	 * @param integer $kpc Keep data count
	 * @param arraykey $rkl Request key name
	 * @param mixed $param Deliver data to each event and delegate callback
	 * @return array Request keep data
	 */
	private function _getKR( $kpc, $rkl, $param )
	{
		if( $kpc < 0 ){
			throw new UnexpectedValueException;
		}

		$kprr = YggRawWebRequest::getRequest();
		foreach( $rkl as $key ){
			unset( $kprr[$key] );
		}

		$kpr = array();
		$cnt = 0;
		foreach( $kprr as $key => $val ){
			try{
				$kpr[$key] = $this->dgStable->exec( $val, $param );
			}
			catch( Exception $e ){
				continue;
			}

			$cnt++;
			if( $cnt >= $kpc ){
				break;
			}
		}

		return $kpr;
	}
}
?>
