<?php
/**
 * Mail format creation class
 *
 * This file defines YggMailFormatter class.
 * It includes methods to make mail data.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require config file
 */
require_once( "YggDore/Base/Config.php" );
/**
 * Require YggDelegate class
 */
require_once( "YggDore/Base/YggDelegate.php" );
/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );


/**
 * Mail format creation class
 *
 * This class has methods to make mail data.
 *
 * <code>
 * // Single part mail
 * $ph = popen( "/usr/sbin/sendmail -t -ihoge@hoge.com", "w" );
 * $mail = new YggMailFormatter();
 * $mail->start( $ph, YggMailFormatter::SINGLEPART );
 * $mail->toList( 
 *     array(
 *         array( "mail" => "hoge@yggdore.com", "name" => "hogehoe" ),
 *         array( "mail" => "test@yggdore.com" )
 *     )
 * );
 * $mail->from( "from@yggdore.com", "YggDore Tester" );
 * $mail->subject( "Title" );
 * $mail->message( "test\ntes." );
 * $mail->end();
 *
 * // Multi part mail
 * $mail = new YggMailFormatter();
 * $mail->start( $ph, YggMailFormatter::MULTIPART );
 * $mail->toList( 
 *     array(
 *         array( "mail" => "hoge@yggdore.com", "name" => "hogehoge" ),
 *         array( "mail" => "test@yggdore.com" )
 *     )
 * );
 * $mail->from( "webmaster@yggdore.com", "YggDore Webmaster" );
 * $mail->subject( "Title" );
 * $mail->boundary();
 * $mail->message( "test\ntes." );
 * $mail->boundary();
 * $mail->attachData( "test.txt", "file data" );
 * $mail->end();
 * </code>
 *
 * @package YggDore.Base
 */
class YggMailFormatter {
	/**
	 * Single part mode
	 *
	 * This value is used by YggMailFormatter::start method.
	 * It shows to make single part mail message.
	 */
	const SINGLEPART = 1;
	/**
	 * Multi part mode
	 *
	 * This value is used by YggMailFormatter::start method.
	 * It shows to make multi part mail message.
	 */
	const MULTIPART  = 2;

	/**
	 * Multi header section
	 *
	 * This value is used by YggMailFormatter::getSection method.
	 * It shows that current processing section is part of multi header.
	 */
	const MHEADSEC  = 1;
	/**
	 * Header section
	 *
	 * This value is used by YggMailFormatter::getSection method.
	 * It shows that current processing section is part of header.
	 */
	const HEADSEC  = 2;
	/**
	 * Body section
	 *
	 * This value is used by YggMailFormatter::getSection method.
	 * It shows that current processing section is part of body.
	 */
	const BODYSEC  = 3;

	/**
	 * Header bytes per line ( includes new line code )
	 *
	 * This value is necessary to specify number of bytes between from 60 to 76.
	 */
	const MIMEHEADER_LINESIZE = 70;
	/**
	 * Maximum bytes of mail address length
	 *
	 * This value is specified less than MIMEHEADER_LINESIZE - 5.
	 * The five characters are <, >, CR, LF, and header space charctor.
	 */
	const MAIL_MAXLEN = 50;


	/**
	 * Send mail mode
	 */
	private $_mode;
	/**
	 * Current section part
	 *
	 * 1: Multipart header<br />
	 * 2: Header<br />
	 * 3: Body
	 */
	private $_sec;
	/**
	 * This is used to check whether current section is empty
	 */
	private $_isse;
	/**
	 * Border ID
	 */
	private $_bid;
	/**
	 * Sending data cache
	 */
	private $_sc;
	/**
	 * Stream handler
	 */
	private $_handle;

	
	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->_clear();
	}


	/**
	 * Clear data
	 *
	 * Member variables of the instance is initialized.
	 */
	private function _clear()
	{
		$this->_mode   = null;
		$this->_sec    = null;
		$this->_isse   = null;
		$this->_bid    = null;
		$this->_sc     = null;
		$this->_handle = null;
	}


	/**
	 * Get send mail mode
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer This value means as follows<br />
	 * YggMailFormatter::SINGLEPART : Single part<br />
	 * YggMailFormatter::MULTIPART : Multi part
	 */
	public function getMode()
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}

		return $this->_mode;
	}


	/**
	 * Get current section part
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer This value means as follows<br />
	 * YggMailFormatter::MHEADSEC : Multi part header<br />
	 * YggMailFormatter::HEADSEC : Header<br />
	 * YggMailFormatter::BODYSEC : Body
	 */
	public function getSection()
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}

		return $this->_sec;
	}


	/**
	 * Get boundary ID
	 *
	 * This method returns boundary ID in the instance.
	 *
	 * @return string
	 */
	public function getBoundary()
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}

		return $this->_bid;
	}


	/**
	 * Write data
	 *
	 * Writes $data to stream.
	 *
	 * @param string $data Target data
	 */
	protected function _write( $data )
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}

		if( $data == "" ){
			throw new UnexpectedValueException;
		}

		$rts = fwrite( $this->_handle, $data );
		if( $rts === false ){
			throw new RuntimeException;
		}

		// Save cache variable data is written to steam.
		// Because it is used to find boundary id.
		$idx = strlen( $data ) - ( strlen( $this->_bid ) - 1 );
		if( $idx < 0 ){
			$this->_sc = substr( $this->_sc, $idx ) . $data;
		}
		else{
			$this->_sc = substr( $data, $idx );
		}
	}


	/**
	 * Start mail creation
	 *
	 * This method begins to create mail data.<br />
	 * You must call YggMailFormatter::end method when
	 * you finishes to use the instance.
	 *
	 * @param resource $handle Stream hander
	 * @param integer $mode Part mode<br />
	 * YggMailFormatter::SINGLEPART : Single part mode<br />
	 * YggMailFormatter::MULTIPART : Multi part mode
	 */
	public function start( $handle, $mode = self::SINGLEPART )
	{
		if( $this->_mode !== null ){
			throw new BadMethodCallException;
		}

		if( $mode != self::SINGLEPART && $mode != self::MULTIPART ){
			throw new UnexpectedValueException;
		}

		$bid = YggTool::uniqueID();

		$data = "MIME-Version: 1.0";
		if( $this->_mode == self::MULTIPART ){
			$data .= YGG_EOL .
			         "Content-Type: Multipart/Mixed;" . YGG_EOL .
					 " boundary=\"" . $bid . "\"";
		}

		$this->_mode   = $mode;
		$this->_sec    = self::HEADSEC;
		$this->_isse   = true;
		$this->_bid    = "----=_NextPart_" . $bid;
		$this->_sc     = "";
		$this->_handle = $handle;
		if( $this->_mode == self::MULTIPART ){
			$this->_sec = self::MHEADSEC;
		}

		try{
			$this->_write( $data );
			$this->_isse = false;
		}
		catch( Exception $e ){
			$this->_clear();
			throw $e;
		}
	}


	/**
	 * Finish to create mail data
	 *
	 * This method finishes to create mail data.
	 */
	public function end()
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec != self::BODYSEC ){
			throw new BadMethodCallException;
		}
		elseif( $this->_isse ){
			throw new BadMethodCallException;
		}

		if( $this->_mode == self::MULTIPART ){
			$this->_write( YGG_EOL . "--" . $this->_bid . "--" );
		}

		$this->_clear();
	}


	/**
	 * Write header
	 *
	 * This method writes header to stream.<br />
	 * If current section is body, this method throws Exception.<br />
	 * If $isnh is specified TRUE, $data is written after writting new line code,
	 * otherwise new line code doesn't write. but this may be security hole.<br />
	 * If $iscb is specified TRUE, boundary ID is checked to find in $data.
	 * Its processing is heavy when big data is set.
	 *
	 * @param string $data Writing data
	 * @param boolean $isnh New line header mode
	 * @param boolean $iscb Checking boundary id mode
	 */
	protected function _writeHeader( $data, $isnh = true, $iscb = true )
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec == self::BODYSEC ){
			// Error occurs because this method can call when current section isn't body
			throw new BadMethodCallException;
		}

		$data = YggString::eol( $data, -1 );
		if( $data == "" ){
			throw new UnexpectedValueException;
		}

		$nh = "";
		if( $isnh ){
			$nh = YGG_EOL;
		}

		// When two new line code includes in $data,
		// Section of header finished.
		if( strpos( $nh . $data, YGG_EOL . YGG_EOL ) !== false ){
			throw new UnexpectedValueException;
		}
		
		$endnl = substr( $data, strlen(YGG_EOL) * -1 );
		if( $endnl == YGG_EOL ){
			$data = substr( $data, 0, strlen(YGG_EOL) * -1 );
		}

		// Mail format breaks when boundary ID includes in $data.
		if( $iscb && $this->_mode == self::MULTIPART ){
			$rts = strpos(
				$this->_sc . $nh . $data, "--" . $this->_bid
			);
			if( $rts !== false ){
				throw new RuntimeException;
			}
		}

		$this->_write( $nh . $data );
		$this->_isse = false;
	}


	/**
	 * Write header
	 *
	 * This method writes header to stream.<br />
	 * If current section is body, this method throws Exception.<br />
	 * If $isnh is specified TRUE, $data is written after writting new line code,
	 * otherwise new line code doesn't write. but this may be security hole.
	 *
	 * @param string $data Writing data
	 * @param boolean $isnh New line header mode
	 */
	public function header( $data, $isnh = true )
	{
		$this->_writeHeader( $data, $isnh );
	}


	/**
	 * Check whether current section is header of top
	 *
	 * This method returns TRUE when current section is header part.
	 * Top header means multi part header is Multi header part only, no header part.
	 *
	 * @return boolean Returns TRUE if current section is first header, otherwise returns FALSE
	 */
	protected function _isTopHeader()
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}

		$top = self::HEADSEC;
		if( $this->_mode == self::MULTIPART ){
			$top = self::MHEADSEC;
		}

		if( $this->_sec != $top ){
			return false;
		}
		return true;
	}


	/**
	 * Write body
	 *
	 * This method writes body to stream.<br />
	 * If current section is head part, this method throws Exception.<br />
	 * If $iscb is specified TRUE, boundary ID is checked to find in $data.
	 * Its processing is heavy when big data is set.
	 *
	 * @param string $data Writing data
	 * @param boolean $iscb Checking boundary id mode
	 */
	protected function _writeBody( $data, $iscb = true )
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec == self::MHEADSEC ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec == self::HEADSEC && $this->_isse ){
			throw new BadMethodCallException;
		}

		if( $data == "" ){
			throw new UnexpectedValueException;
		}
		elseif( $iscb && $this->_mode == self::MULTIPART ){
			$rts = strpos( $this->_sc . $data, "--" . $this->_bid );
			if( $rts !== false ){
				throw new RuntimeException;
			}
		}

		if( $this->_sec == self::HEADSEC ){
			$this->_write( YGG_EOL . YGG_EOL );
		}

		$this->_write( $data );

		$this->_sec  = self::BODYSEC;
		$this->_isse = false;
	}


	/**
	 * Write body
	 *
	 * This method writes body to stream.<br />
	 * If current section is head part, this method throws Exception.
	 *
	 * @param string $data Writing data
	 */
	public function body( $data )
	{
		$this->_writeBody( $data );
	}


	/**
	 * Write boundary
	 *
	 * This method writes boundary ID to stream.<br />
	 * This method can call multipart mode and
	 * current section is body part or header of multi part only.
	 */
	public function boundary()
	{
		if( $this->_mode != self::MULTIPART ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec == self::HEADSEC ){
			throw new BadMethodCallException;
		}
		elseif( $this->_isse ){
			throw new BadMethodCallException;
		}

		// Finish current part.
		$eol = YGG_EOL;
		if( $this->_sec == self::MHEADSEC ){
			$eol .= YGG_EOL;
		}

		$this->_write( $eol . "--" . $this->_bid );

		$this->_sec  = self::HEADSEC;
		$this->_isse = true;
	}


	/**
	 * Write address
	 *
	 * This method writes address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : Name (option)<br />
	 * mail => string : Mail address
	 * @param string $head Mail header
	 */
	protected function _writeAddrList( $data, $head )
	{
		if( !$this->_isTopHeader() ){
			throw new BadMethodCallException;
		}

		if( $head == "" ){
			throw new UnexpectedValueException;
		}

		$rts = YggTool::callVirtualStatic(
			$this, "createAddrList", array( $data, $head )
		);

		$this->_writeHeader( $rts, true, false );
	}


	/**
	 * Write from header
	 *
	 * This method writes from header.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param string $addr Mail address
	 * @param string $name Name (option)
	 */
	public function from( $addr, $name = "" )
	{
		$this->_writeAddrList(
			array( array( "name" => $name, "mail" => $addr ) ),
			"From: "
		);
	}


	/**
	 * Write "TO" address
	 *
	 * This method writes "TO" address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : Name (option)<br />
	 * mail => string : Mail address
	 */
	public function toList( $data )
	{
		$this->_writeAddrList( $data, "To: " );
	}


	/**
	 * Write "CC" address
	 *
	 * This method writes "CC" address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : Name (option)<br />
	 * mail => string : Mail address
	 */
	public function ccList( $data )
	{
		$this->_writeAddrList( $data, "CC: " );
	}


	/**
	 * Write "BCC" address
	 *
	 * This method writes "BCC" address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : Name (option)<br />
	 * mail => string : Mail address
	 */
	public function bccList( $data )
	{
		$this->_writeAddrList( $data, "BCC: " );
	}


	/**
	 * Write subject
	 *
	 * This method writes subject in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param string $subject
	 */
	public function subject( $subject )
	{
		if( !$this->_isTopHeader() ){
			throw new BadMethodCallException;
		}

		$subhead = YggTool::callVirtualStatic(
			$this, "createSubjectHeader", array( $subject )
		);

		$this->_writeHeader( $subhead );
	}


	/**
	 * Write message
	 *
	 * This method writes message in header section.<br />
	 * This method can be called when current section is header part.
	 *
	 * @param string $msg
	 */
	public function message( $msg )
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec != self::HEADSEC ){
			throw new BadMethodCallException;
		}

		$msg = YggTool::callVirtualStatic(
			$this, "createMessage", array( $msg )
		);

		$this->_writeHeader(
			YggTool::callVirtualStatic(
				$this, "createMessageHeader"
			)
		);

		$this->_writeBody( $msg, true );
	}


	/**
	 * Write attach data
	 *
	 * This method writes attach data to head and body part.<br />
	 * This method can be called when send mail mode is multipart.
	 *
	 * @param string $afh Attach file header
	 * @param string $data Attach data
	 */
	protected function _writeAttachData( $afh, $data )
	{
		if( $this->_mode === null ){
			throw new BadMethodCallException;
		}
		elseif( $this->_mode != self::MULTIPART ){
			throw new BadMethodCallException;
		}
		elseif( $this->_sec != self::HEADSEC ){
			throw new BadMethodCallException;
		}

		if( $afh == "" ){
			throw new UnexpectedValueException;
		}

		$data = self::createAttachData( $data );

		$this->_writeHeader( $afh );

		// Data isn't checked whether there is boundary ID.
		// Because $data is converted to base64 format.
		$this->_writeBody( $data, false );
	}


	/**
	 * Write attach data
	 *
	 * This method writes attach data whose file name is $fname to stream.<br />
	 * This method can be called when send mail mode is multipart.
	 *
	 * @param string $fname File name
	 * @param string $data Attach data
	 */
	public function attachData( $fname, $data )
	{
		$fhead = YggTool::callVirtualStatic(
			$this, "createAttachFileHeader", array( $fname )
		);

		$this->_writeAttachData( $fhead, $data );
	}


	/**
	 * Write attach data from file stream
	 *
	 * This method writes data that is specified by $handle that is file stream.<br />
	 * $rrc is specified number of line to read once from stream.
	 * One line length is 72 bytes.
	 *
	 * @param string $afh Attach file header
	 * @param resource $handle File stream
	 * @param integer $rrc Number of line to read once
	 */
	protected function _writeAttachFile( $afh, $handle, $rrc = 10 )
	{
		if( $afh == "" ){
			throw new UnexpectedValueException;
		}
		elseif( $rrc <= 0 ){
			throw new RangeException;
		}

		$this->_writeHeader( $afh );

		$ret  = "";
		$rrsz = self::_getAttachDataLineSize();
		$rrsz = ((int)($rrsz / 4)) * 3;
		while( !feof($handle) ){
			$data = stream_get_line( $handle, $rrsz );
			if( $data === false ){
				throw new RuntimeException;
			}
			elseif( $data == "" ){
				continue;
			}

			$data = self::createAttachData( $data );

			// Data isn't checked for looking up boundary id.
			// Because $data is converted to base64.
			$this->_writeBody( $ret . $data, false );

			$ret = YGG_EOL;
		}
		if( $ret == "" ){
			throw new UnexpectedValueException;
		}
	}


	/**
	 * Write attach data from file stream
	 *
	 * This method writes data that is specified by $handle that is file stream.<br />
	 * The stream data is named as $fname.<br />
	 * $rrc is specified number of line to read once from stream.
	 * One line length is 72 bytes.
	 *
	 * @param string $fname File name
	 * @param resource $handle File stream
	 * @param integer $rrc Read line count once
	 */
	public function attachFile( $fname, $handle, $rrc = 10 )
	{
		$fhead = self::createAttachFileHeader( $fname );
		$this->_writeAttachFile( $fhead, $handle, $rrc );
	}


	/**
	 * Check mail address
	 *
	 * This method checks that $mailaddr is global mail.
	 *
	 * @param string $mailaddr
	 * @return boolean Returns TRUE if $mailaddr is global mail address,
	 * otherwise returns FALSE.
	 */
	public static function checkGlobalAddr( $mailaddr )
	{
		$chkptn = '/^' .
		          '[0-9a-z_\.\-]+@' .
		          '[0-9a-z_\-]+\.[0-9a-z_\.\-]+' .
				  '$/i';

		if( $mailaddr == "" ){
			return false;
		}
		elseif( strlen( $mailaddr ) > self::MAIL_MAXLEN ){
			return false;
		}
		elseif( !preg_match( $chkptn, $mailaddr ) ){
			return false;
		}
		elseif( preg_match( '/@.*\.\./', $mailaddr ) ){
			return false;
		}
		elseif( preg_match( '/@127\.0\.0\.1/', $mailaddr ) ){
			return false;
		}

		// If the last character is new line code, error occurs.
		// Add processing because old php version doesn't occur error.
		$lst = substr( $mailaddr, -1 );
		if( !preg_match( '/^[a-z]/i', $lst ) ){
			return false;
		}

		return true;
	}


	/**
	 * Create mail address list
	 *
	 * This method creates mail address from $data.<br />
	 * Mail format is [name <maddr>] if you specify name,
	 * otherwise mail format is [maddr].<br />
	 * Maximum length of mail address is YggMailFormatter::MAIL_MAXLEN.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : name (option)<br />
	 * mail => string : mail address
	 * @return string
	 */
	public static function createAddrList( $data, $head = "" )
	{
		$mxlsz = self::MIMEHEADER_LINESIZE - strlen( YGG_EOL );
		
		if( count( $data ) <= 0 ){
			throw new UnexpectedValueException;
		}
		elseif( strlen( $head ) > $mxlsz ){
			throw new UnexpectedValueException;
		}

		$rts   = $head;
		$lsz   = strlen( $head );
		$spsp  = "";
		$keyl  = array_keys( $data );
		$keylc = count( $keyl );
		for( $i = 0; $i < $keylc; $i++ ){
			$rec =& $data[$keyl[$i]];

			$isnm = false;
			if( isset( $rec['name'] ) && $rec['name'] != "" ){
				$isnm = true;
			}

			if( $isnm ){
				if( strlen( $rec['name'] ) >= $mxlsz ){
					throw new UnexpectedValueException;
				}
				elseif( !self::_checkEOL( $rec['name'] ) ){
					throw new UnexpectedValueException;
				}
				elseif( preg_match( '/[<>,]/', $rec['name'] ) ){
					throw new UnexpectedValueException;
				}
			}

			if( !self::checkGlobalAddr( $rec['mail'] ) ){
				throw new UnexpectedValueException;
			}

			if( $isnm ){
				$wsz = $lsz + strlen( $spsp . $rec['name'] );

				if( $wsz > $mxlsz ){
					// Name and email address writes two lines
					$rts .= YGG_EOL;
					$lsz  = 0;
					$spsp = " ";
				}

				$w = $spsp . $rec['name'];

				$rts  .= $w;
				$lsz  += strlen( $w ); 

				$spsp  = " ";
			}

			// Set comma if there is next record.
			$csp = "";
			if( $i < $keylc - 1 ){
				$csp = ",";
			}

			$w    = "<" . $rec['mail'] . ">" . $csp;
			$wsz  = $lsz + strlen( $spsp . $w );
			if( $wsz > $mxlsz ){
				// Name and email address writes two lines
				$rts .= YGG_EOL;
				$lsz  = 0;
				$spsp = " ";
			}

			$w     = $spsp . $w;
			$rts  .= $w;
			$lsz  += strlen( $w );

			$spsp  = " ";
		}

		return $rts;
	}


	/**
	 * Check new line
	 *
	 * This method checks that new line code exists in $str.
	 *
	 * @param string $str Target string
	 * @return boolean Returns TRUE if there is new line code in $str,
	 * otherwize returns FALSE
	 */
	protected static function _checkEOL( $str )
	{
		$str = YggString::eol( $str, -1 );
		if( $str == "" ){
			return false;
		}
		elseif( strpos( $str, YGG_EOL ) !== false ){
			return false;
		}

		return true;
	}


	/**
	 * Create subject header
	 *
	 * This method creates subject header from $subject.
	 *
	 * @param string $subject
	 * @return string
	 */
	public static function createSubjectHeader( $subject )
	{
		$rts = self::_checkEOL( $subject );
		if( !$rts ){
			throw new UnexpectedValueException;
		}

		$mxlsz = self::MIMEHEADER_LINESIZE - strlen( YGG_EOL );

		$head = "Subject: ";
		$hlen = strlen( $head );
		if( $hlen + strlen( $subject ) > $mxlsz ){
			throw new UnexpectedValueException;
		}

		return $head . $subject;
	}


	/**
	 * Create mail message header
	 *
	 * This method returns mail message header.
	 *
	 * @return string
	 */
	public static function createMessageHeader()
	{
		return "Content-Type: text/plain; charset=us-ascii";
	}


	/**
	 * Create mail message
	 *
	 * This method adjusts new line code of $msg and returns it.
	 *
	 * @param string $msg Target of message
	 * @return string
	 */
	public static function createMessage( $msg )
	{
		$msg = YggString::eol( $msg, -1 );
		if( $msg == "" ){
			throw new UnexpectedValueException;
		}

		return $msg;
	}


	/**
	 * Create attach data header
	 *
	 * Create attaching header that has attach file name $fname.
	 *
	 * @param string $fname File name
	 * @return string
	 */
	protected static function _createAttachFileHeader( $fname )
	{
		$rts = self::_checkEOL( $fname );
		if( !$rts ){
			throw new UnexpectedValueException;
		}
		elseif( strpos( $fname, "\"" ) !== false ){
			throw new UnexpectedValueException;
		}

		$data = array( 
			"Content-Type: application/octet-stream;",
			" name=\"%fname%\"",
			"Content-Disposition: attachment;",
			" filename=\"%fname%\"",
			"Content-Transfer-Encoding: base64"
		);

		$mxlsz = self::MIMEHEADER_LINESIZE - strlen( YGG_EOL );

		$rts = "";
		$sp  = "";
		for( $i = 0; $i < count($data); $i++ ){
			$w = YggString::embed(
				$data[$i], array( "fname" => $fname )
			);

			if( strlen( $w ) > $mxlsz ){
				throw new UnexpectedValueException;
			}

			$rts .= $sp . $w;
			
			$sp = YGG_EOL;
		}

		return $rts;
	}


	/**
	 * Create attach data header
	 *
	 * Create attaching header that has attach file name $fname.
	 *
	 * @param string $fname File name
	 * @return string
	 */
	public static function createAttachFileHeader( $fname )
	{
		return self::_createAttachFileHeader( $fname );
	}


	/**
	 * Get attach data line length size
	 *
	 * This method returns one line size that converts to base64.<br />
	 * This size is four multiples because base64 size is four multiples.
	 *
	 * @return integer
	 */
	protected static function _getAttachDataLineSize()
	{
		$lsz = self::MIMEHEADER_LINESIZE - strlen( YGG_EOL );

		// Get four multiples because base64 size is four multiples
		return ((int)( $lsz / 4 )) * 4;
	}


	/**
	 * Create attach data
	 *
	 * This method converts $data to base64 format and return it.
	 *
	 * @param string $data Target string
	 * @return string
	 */
	public static function createAttachData( $data )
	{
		if( $data == "" ){
			throw new UnexpectedValueException;
		}

		return YggString::trim(
			chunk_split(
				base64_encode( $data ),
				self::_getAttachDataLineSize(),
				YGG_EOL
			)
		);
	}
}
?>
