<?php
/**
 * Mysqli class
 *
 * This file defines YggMySQLi class.
 * It includes methods to create sql query for mysql and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggSafe file
 */
require_once( "YggDore/Base/YggSafe.php" );
/**
 * Require YggInt file
 */
require_once( "YggDore/Base/YggInt.php" );
/**
 * Require YggNumeric file
 */
require_once( "YggDore/Base/YggNumeric.php" );
/**
 * Require YggString file
 */
require_once( "YggDore/Base/YggString.php" );


/**
 * Mysqli class
 *
 * This class has methods to create sql query method for mysql and so on.
 * This class extends MySQLi class.
 *
 * <code>
 * $mysqli = new YggMySQLi("localhost", "my_user", "my_password", "world");
 * $rts1 = $mysqli->sql_string( "aaa" );
 * $rts2 = $mysqli->sql_string( "" );
 * $rts3 = $mysqli->sql_string( null );
 * $mysqli->close();
 *
 * (Result)
 * $rts1 => "'aaa'";
 * $rts2 => "''";
 * $rts3 => "null";
 * </code>
 *
 * @package YggDore.Base
 */
class YggMySQLi extends MySQLi {
	/**
	 * No convert
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data isn't converted.
	 */
	const NOESC = 1;
	/**
	 * Numeric convert
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data is converted to safety numeric.
	 */
	const NUMERIC = 2;
	/**
	 * Integer convert
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data is converted to safety integer.
	 */
	const INT = 3;
	/**
	 * String convert ( enclose by single quotation )
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data is converted to safety string and enclosed by single quotation.
	 */
	const STRING = 4;
	/**
	 * String convert ( Not enclose by single quotation )
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data is converted to string and escape.
	 */
	const ESCSTR = 5;
	/**
	 * Boolean convert
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data is converted to boolean.
	 */
	const BOOL = 6;
	/**
	 * Field and table name convert
	 *
	 * This value is used by YggMySQLi::sql_data and so on.
	 * It shows that the target data is converted to safety field and table name.
	 */
	const FT = 7;


	/**
	 * Convert to type of string
	 *
	 * $str is enclosed by single quotation and escaped safety.<br />
	 * This method returns NULL string if $str is specified NULL value.
	 *
	 * @param mixed $str Target value
	 * @return string
	 */
	public function sql_string( $str )
	{
		if( $str === null ){
			return $this->real_escape_string( "null" );
		}

		$str = YggSafe::safeString( $str );
		$str = $this->real_escape_string( $str );
		return "'" . $str . "'";
	}


	/**
	 * Convert to type of numeric
	 *
	 * $str is converted to type of numeric.<br />
	 * This method throws exception if $str isn't numeric.<br />
	 * This method returns NULL string if $str is specified NULL value.
	 *
	 * @param mixed $str Target value
	 * @return string
	 */
	public function sql_numeric( $str )
	{
		if( $str === null ){
			return $this->real_escape_string( "null" );
		}

		$str = YggSafe::safeNumeric( $str );
		$str = YggNumeric::conv( $str );
		return $this->real_escape_string( $str );
	}


	/**
	 * Convert to type of integer
	 *
	 * $str is converted to type of integer.<br />
	 * This method throws exception if $str isn't integer.<br />
	 * This method returns NULL string if $str is specified NULL value.
	 *
	 * @param mixed $str Target value
	 * @return string
	 */
	public function sql_int( $str )
	{
		if( $str === null ){
			return $this->real_escape_string( "null" );
		}

		$str = YggSafe::safeNumeric( $str );
		$str = YggInt::conv( $str );
		return $this->real_escape_string( $str );
	}


	/**
	 * Convert to type of boolean
	 *
	 * $str is converted to type of boolean.<br />
	 * This method returns NULL string if $str is specified NULL value.
	 *
	 * @param mixed $str Target value
	 * @return string
	 */
	public function sql_bool( $str )
	{
		if( $str === null ){
			return $this->real_escape_string( "null" );
		}

		$str = YggSafe::safeBool( $str );
		if( !$str ){
			return $this->real_escape_string( "0" );
		}

		return $this->real_escape_string( "1" );
	}


	/**
	 * Convert to field and table name
	 *
	 * $str is converted to safety field and table name.<br />
	 * This method throws exception if $str has backslash and be empty.<br />
	 * This method returns NULL string if $str is specified NULL value.
	 *
	 * @param mixed $str Target value
	 * @return string
	 */
	public function sql_ft( $str )
	{
		if( $str == "" ){
			throw new UnexpectedValueException;
		}
		elseif( strpos( $str, "`" ) !== false ){
			throw new UnexpectedValueException;
		}

		return '`' . $str . '`';
	}


	/**
	 * Create sql data
	 *
	 * This method converts $data to sql data.<br />
	 * This method throws exception when error occurs.
	 *
	 * @param array $src Target data
	 * @param array $sit Convert information<br />
	 * Key of its elements is key name in $data. Value is one of the following data.<br />
	 * YggMySQLi::NOESC   : Not convert<br />
	 * YggMySQLi::NUMERIC : Numeric<br />
	 * YggMySQLi::INT     : Integer numeric<br />
	 * YggMySQLi::STRING  : String (add single quotation)<br />
	 * YggMySQLi::ESCSTR  : String (without single quotation)<br />
	 * YggMySQLi::BOOL    : Boolean<br />
	 * YggMySQLi::FT      : Field and table name
	 * @return array
	 */
	public function sql_data( $src, $sit )
	{
		$sqld = array();

		if( count($sit) <= 0 ){
			throw new UnexpectedValueException;
		}

		$exctbl = array(
			self::NOESC   => "",
			self::ESCSTR  => "real_escape_string",
			self::STRING  => "sql_string",
			self::INT     => "sql_int",
			self::NUMERIC => "sql_numeric",
			self::BOOL    => "sql_bool",
			self::FT      => "sql_ft"
		);

		foreach( $sit as $skey => $mode ){
			if( $mode < self::NOESC || $mode > self::FT ){
				throw new UnexpectedValueException;
			}

			if( !isset($src[$skey]) ){
				$sqld[$skey] = null;
			}

			if( $exctbl[$mode] == "" ){
				$sqld[$skey] = $src[$skey];
				continue;
			}
			$sqld[$skey] = call_user_func(
				array( $this, $exctbl[$mode] ), $src[$skey]
			);
		}

		return $sqld;
	}


	/**
	 * Create sql query
	 *
	 * This method converts $data to sql data and
	 * embedded them to sql query specified by $sql.
	 * Embed keyword is each keys name of $sit.<br />
	 * This method throws exception when error occurs.
	 *
	 * @param string $sql Query string
	 * @param array $src Same as YggMySQLi::sql_data
	 * @param array $sit Same as YggMySQLi::sql_data
	 * @return string Query
	 */
	public function sql( $sql, $src, $sit )
	{
		if( $sql == "" ){
			throw new UnexpectedValueException;
		}

		$rts = $this->sql_data( $src, $sit );

		return YggString::embed( $sql, $rts );
	}


	/**
	 * Create multi sql query
	 *
	 * This method reads data in $sqlary sequentially and semicolon is inserted among them.<br />
	 * This method converts $data to sql data and embedded them to sql query.
	 * Embed keyword is each keys name of $sit.<br />
	 * This method throws exception when error occurs.
	 *
	 * @param string $sqlary Query string array
	 * @param array $src Same as YggMySQLi::sql_data
	 * @param array $sit Same as YggMySQLi::sql_data
	 * @return string Query
	 */
	public function sqls( $sqlary, $src, $sit )
	{
		$sa  = $this->sql_array( $sqlary, $src, $sit );
		$rts = "";
		$sp  = "";
		foreach( $sa as $sql ){
			$rts .= $sp . $sql;
			$sp   = "; ";
		}

		return $rts;
	}


	/**
	 * Create multi sql query (Returns array version)
	 *
	 * This method reads data in $sqlary sequentially and
	 * $data is converted to sql data and embedded them to each sql queries.
	 * Embed keyword is each keys name of $sit.<br />
	 * This method throws exception when error occurs.
	 *
	 * @param string $sqlary Query string array
	 * @param array $src Same as YggMySQLi::sql_data
	 * @param array $sit Same as YggMySQLi::sql_data
	 * @return array Result sql query array
	 */
	public function sql_array( $sqlary, $src, $sit )
	{
		if( count( $sqlary ) <= 0 ){
			return null;
		}

		foreach( $sqlary as $key => $sql ){
			$sqlary[$key] = $this->sql( $sql, $src, $sit );
		}

		return $sqlary;
	}
	

	/**
	 * Create updating query
	 *
	 * This method converts $data to sql data and
	 * creates sql query of "key1=val1, key2=val2 ... keyn=valn" format.
	 *
	 * @param array $data Target data
	 * @param array $sit Same as YggMySQLi::sql_data
	 * @return string Update query
	 */
	public function sql_update_list( $data, $sit )
	{
		$sqld = $this->sql_data( $data, $sit );

		$sql = "";
		$sp  = "";
		foreach( $sqld as $key => $val ){
			$key = $this->sql_ft( $key );
			if( $val == "" ){
				throw new UnexpectedValueException;
			}

			$sql .= $sp .
					$key .
					" = " . 
					$val;
			$sp   = ", ";
		}

		return $sql;
	}
	

	/**
	 * Split keyword and create sql query
	 *
	 * $wsstr splits keywords per space character and
	 * sql query is created from $sql, $sep and $mode arguments.<br />
	 * If sql query of $sql has "%word%" code, it replaces to word.
	 * It is made per number of keywords. $sep is added among sql queries.<br />
	 * $mode is specified type of keyword.
	 *
	 * @param string $wsstr Target string
	 * @param string $sql Query string
	 * @param string $sep Separate word of keyword
	 * @param integer $mode Same as $sit of YggMySQLi::sql_data
	 * @return string
	 */
	public function sql_word_split( $wsstr, $sql, $sep = "AND", $mode = self::STRING )
	{
		if( $sql == "" ){
			throw new UnexpectedValueException;
		}
		elseif( strpos( $sql, "%word%" ) === false ){
			throw new UnexpectedValueException;
		}
		elseif( $wsstr == "" ){
			return "";
		}

		$query = "";
		$sp    = "";
		if( $sep != "" ){
			$sep = " " . $sep;
		}

		$wordary = preg_split( '/\s+/', $wsstr );
		if( !is_array($wordary) || count($wordary) <= 0 ){
			throw new UnexpectedValueException;
		}

		foreach( $wordary as $word ){
			if( $word == "" ){
				continue;
			}

			$w = $this->sql_data(
				array( "word" => $word ), array( "word" => $mode )
			);

			$query .= $sp;
			$sp     = $sep . " ";

			$query .= YggString::embed( $sql, $w );
		}

		return $query;
	}
	

	/**
	 * Close rest result set
	 *
	 * If you call multi_query function and so on, you gets multi results set.
	 * This method releases them.
	 */
	public function multi_free()
	{
		while( $this->more_results() ){
			$rts = $this->next_result();
			if( !$rts ){
				throw new RuntimeException;
			}

			if( $this->field_count <= 0 ){
				continue;
			}

			$result = $this->use_result();
			if( !$result ){
				throw new RuntimeException;
			}
			$result->free();
		}
	}
}
?>
