<?php
/**
 * Safety value and type creation class
 *
 * This file defines YggSafe class.
 * It includes methods to convert value safety.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggInt class
 */
require_once( "YggDore/Base/YggInt.php" );


/**
 * Safety value and type creation class
 *
 * This class has methods to convert value safety.
 *
 * @package YggDore.Base
 */
class YggSafe {
	/**
	 * Convert safety boolean value
	 *
	 * This method converts $data to type of boolean.<br />
	 * If type of $data isn't NULL, boolean, numeric and string,
	 * UnexpectedValueException is thrown.
	 *
	 * <code>
	 * $data = null;
	 * $rts  = YggSafe::safeBool( $data );
	 * (Result)
	 * $rts => 0
	 * </code>
	 * 
	 * @param mixed $data Target value
	 * @return boolean
	 */
	public static function safeBool( $data )
	{
		if( !is_scalar( $data ) && !is_null( $data ) ){
			throw new UnexpectedValueException;
		}

		return (bool)$data;
	}


	/**
	 * Convert safety string value
	 *
	 * This method converts $data to type of string.<br />
	 * This method returns "1"(string) when $data is TRUE(boolean),
	 * and returns "0"(string) when $data is FALSE(boolean)
	 * because YggSafe::Numeric method returns them.<br />
	 * If type of $data isn't NULL, boolean, numeric and string,
	 * UnexpectedValueException is thrown.
	 *
	 * <code>
	 * $data = 15;
	 * $rts  = YggSafe::safeString( $data );
	 * (Result)
	 * $rts => "15"
	 * </code>
	 * 
	 * @param mixed $data Target value
	 * @return string
	 */
	public static function safeString( $data )
	{
		if( is_string( $data ) ){
			return $data;
		}
		elseif( is_null( $data ) ){
			return "";
		}
		elseif( is_int( $data ) || is_float( $data ) ){
			return (string)$data;
		}
		elseif( !is_bool( $data ) ){
			throw new UnexpectedValueException;
		}

		return (string)(int)$data;
	}


	/**
	 * Convert safety numeric value
	 *
	 * This method converts $data to type of numeric.<br />
	 * this method returns type of integer if result of converting is integer value,
	 * otherwise returns type of float.<br />
	 * This method returns the following value.<br />
	 * TRUE (boolean) => 1 (integer)<br />
	 * FALSE (boolean) => 0 (integer)<br />
	 * NULL (NULL) => 0 (integer)<br />
	 * UnexpectedValueException is thrown when converting failed.
	 *
	 * <code>
	 * $data = "15";
	 * $rts  = YggSafe::safeNumeric( $data );
	 * (Result)
	 * $rts => 15
	 * </code>
	 * 
	 * @param mixed $data Target value
	 * @return numeric
	 */
	public static function safeNumeric( $data )
	{
		if( is_int( $data ) ){
			return $data;
		}
		elseif( is_float( $data ) ){
			if( YggInt::is( $data ) ){
				return (int)$data;
			}

			return $data;
		}
		elseif( is_numeric( $data ) ){
			$data = (float)$data;
			// If can convert $data to integer, returns it.
			if( YggInt::is( $data ) ){
				return (int)$data;
			}

			return $data;
		}
		elseif( is_bool( $data ) ){
			return (int)$data;
		}
		elseif( !is_null( $data ) ){
			throw new UnexpectedValueException;
		}

		return 0;
	}


	/**
	 * Convert safety integer value
	 *
	 * This method converts $data to type of integer.<br />
	 * If converting from numeric to integer fails, UnexpectedValueException is thrown.
	 *
	 * <code>
	 * $data = null;
	 * $rts  = YggSafe::safeInt( $data );
	 * (Result)
	 * $rts => 0
	 * </code>
	 * 
	 * @param mixed $data Target value
	 * @return integer
	 */
	public static function safeInt( $data )
	{
		$data = YggSafe::safeNumeric( $data );
		if( !is_int( $data ) ){
			throw new UnexpectedValueException;
		}

		return $data;
	}


	/**
	 * Convert safety array value
	 *
	 * This method converts $data to type of array.<br />
	 * If type of $data isn't array, UnexpectedValueException is thrown.
	 *
	 * <code>
	 * $data = array();
	 * $rts  = YggSafe::safeArray( $data );
	 * (Result)
	 * $rts => array()
	 * </code>
	 * 
	 * @param mixed $data Target value
	 * @return array
	 */
	public static function safeArray( $data )
	{
		if( is_array( $data ) ){
			return $data;
		}
		elseif( !is_null( $data ) ){
			throw new UnexpectedValueException;
		}

		return array();
	}


	/**
	 * Convert safety array key
	 *
	 * This method converts $data to type of array key.<br />
	 * Array key type defines integer or string in PHP manual.<br />
	 * This method returns the following value.<br />
	 * 10 (integer) => 10 (integer)<br />
	 * 10.5 (float) => "10.5" (string)<br />
	 * "string" (string) => "string" (string)<br />
	 * TRUE (boolean) => 1 (integer)<br />
	 * FALSE (boolean) => 0 (integer)<br />
	 * NULL (NULL) => "" (string)<br />
	 * If converting from numeric to integer fails, UnexpectedValueException is thrown.
	 *
	 * <code>
	 * $data = null;
	 * $rts  = YggSafe::safeArrayKey( $data );
	 * (Result)
	 * $rts => ""
	 * </code>
	 * 
	 * @param mixed $data Target value
	 * @return integer_or_string
	 */
	public static function safeArrayKey( $data )
	{
		if( is_string($data) || is_int($data) ){
			return $data;
		}
		elseif( is_null($data) ){
			return "";
		}
		elseif( is_bool($data) ){
			return (int)$data;
		}
		elseif( !is_float($data) ){
			throw new UnexpectedValueException;
		}

		return (string)$data;
	}
}
?>
