<?php
/**
 * Sendmail class
 *
 * This file defines YggSendMail class.
 * It includes methods to send mail.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require config file
 */
require_once( "YggDore/Base/Config.php" );
/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );
/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );
/**
 * Require YggMailFormatter class
 */
require_once( "YggDore/Base/YggMailFormatter.php" );


/**
 * Sendmail class
 *
 * This class has methods to send mail.
 * When this class transports data, pipe to sendmail is used.
 * It means to be very slow when you want to send a lot of mail.
 *
 * <code>
 * (Normal)
 * $mailobj = YggSendmail::open( YggSendMail::SINGLEPART );
 * $mailobj->toList( 
 *     array(
 *         array( "mail" => "hoge@yggdore.com", "name" => "hogehoge" ),
 *         array( "mail" => "test@yggdore.com" )
 *     )
 * );
 * $mailobj->from( "webmaster@yggdore.com", "YggDore Webmaster" );
 * $mailobj->subject( "Title" );
 * $mailobj->message( "hogehoge.\nhoge." );
 * $mailobj->close();
 *
 * (Attaching test.txt mail)
 * $mailobj = YggSendmail::open( YggSendMail::MULTIPART );
 * $mailobj->toList( 
 *     array(
 *         array( "mail" => "hoge@yggdore.com", "name" => "hogehoge" ),
 *         array( "mail" => "test@yggdore.com" )
 *     )
 * );
 * $mailobj->from( "webmaster@yggdore.com", "YggDore Webmaster" );
 * $mailobj->subject( "Title" );
 * $mailobj->Boundary();
 * $mailobj->message( "hogehoge.\nhoge." );
 * $mailobj->Boundary();
 * $mailobj->attachData( "test.txt", "testdata" );
 * $mailboj->close();
 * </code>
 *
 * @package YggDore.Base
 */
class YggSendMail {
	/**
	 * Single part sending mail mode
	 */
	const SINGLEPART = YggMailFormatter::SINGLEPART;
	/**
	 * Multi part sending mail mode
	 */
	const MULTIPART = YggMailFormatter::MULTIPART;


	/**
	 * Pipe stream handler for sendmail
	 */
	private $_handle;
	/**
	 * Instance of YggMailFormatter class
	 */
	private $_mf;

	
	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct() 
	{
		$this->_handle = null;
		$this->_mf     = null;
	}


	/**
	 * Open pipe to sendmail
	 *
	 * This method opens pipe of sendmail specified by $smpath.<br />
	 * This method opens sendmail with "-t $smopt".
	 * Be careful if you use $smopt for security.<br />
	 * If you want to use Envelope-From, you should call YggSendMail::openFrom method.<br />
	 * $mode is mail sending mode.
	 * YggSendMail::SINGLEPART use to send text only mail.
	 * YggSendMail::MULTIPART use to send text and attaching data mail.<br />
	 * If $mf is specified NULL, YggMailFormatter is used in the instance.<br />
	 * YggSendMail::close method must be called when you finishes to use the instance.
	 *
	 * @param integer $mode Sendmail mode<br />
	 * YggSendMail::SINGLEPART : Send simple text mail<br />
	 * YggSendMail::SINGLEPART : Send text and attaching data mail
	 * @param string $smpath Path of sendmail
	 * @param string $smopt Sendmail option
	 * @param YggMailFormatter $mf Mail format controller
	 * @return YggSendMail
	 */
	public static function open(
		$mode = self::SINGLEPART,
		$smpath = YGG_SENDMAILPATH,
		$smopt = "",
		$mf = null
	)
	{
		$sp = "";
		if( $smopt != "" && substr( $smopt, 0, 1 ) != " " ){
			$sp = " ";
		}

		return self::openCommand( $mode, $smpath, "-t" . $sp . $smopt, $mf );
	}


	/**
	 * Open pipe to sendmail and send Envelope-FROM
	 *
	 * This method opens pipe of sendmail specified by $smpath.<br />
	 * This method opens sendmail with "$smpath -t -f$efrom $smopt".
	 * Be careful if you use $smopt for security.<br />
	 * $mode is that sending mail mode.
	 * YggSendMail::SINGLEPART may use to send text only mail.
	 * YggSendMail::MULTIPART may use to send text and attaching data mail.<br />
	 * If $mf is specified NULL, YggMailFormatter is used in the instance.<br />
	 * YggSendMail::close method must be called when you finishes to use the instance.
	 *
	 * @param string $efrom Envelope-From address
	 * @param integer $mode Sendmail mode<br />
	 * YggSendMail::SINGLEPART : Send simple text mail<br />
	 * YggSendMail::SINGLEPART : Send text and attaching data mail
	 * @param string $smpath Path of sendmail
	 * @param string $smopt Sendmail option
	 * @param YggMailFormatter $mf Mail format controller
	 * @return YggSendMail
	 */
	public static function openEnvelopeFrom(
		$efrom,
		$mode = self::SINGLEPART,
		$smpath = YGG_SENDMAILPATH,
		$smopt = "",
		$mf = null
	)
	{
		if( !YggMailFormatter::checkGlobalAddr( $efrom ) ){
			throw new UnexpectedValueException; 
		}

		$sp = "";
		if( $smopt != "" && substr( $smopt, 0, 1 ) != " " ){
			$sp = " ";
		}

		return self::openCommand( $mode, $smpath, "-t -f" . $efrom . $sp . $smopt, $mf );
	}


	/**
	 * Open pipe to sendmail
	 *
	 * This method opens pipe of sendmail specified by $smpath.<br />
	 * This method opens sendmail with "$smopt".
	 * Be careful if you use $smopt for security.<br />
	 * $mode is mail sending mode.
	 * YggSendMail::SINGLEPART use to send text only mail.
	 * YggSendMail::MULTIPART use to send text and attaching data mail.<br />
	 * If $mf is specified NULL, YggMailFormatter is used in the instance.<br />
	 * YggSendMail::close method must be called when you finishes to use the instance.
	 *
	 * @param integer $mode Sendmail mode<br />
	 * YggSendMail::SINGLEPART : Send simple text mail<br />
	 * YggSendMail::SINGLEPART : Send text and attaching data mail
	 * @param string $smpath Path of sendmail
	 * @param string $smopt Sendmail option
	 * @param YggMailFormatter $mf Mail format controller
	 * @return YggSendMail
	 */
	public static function openCommand(
		$mode = self::SINGLEPART,
		$smpath = YGG_SENDMAILPATH,
		$smopt = "",
		$mf = null
	)
	{
		if( $smpath == "" ){
			throw new UnexpectedValueException; 
		}
		elseif( !is_executable( $smpath ) ){
			throw new UnexpectedValueException; 
		}

		$sp = "";
		if( $smopt != "" && substr( $smopt, 0, 1 ) != " " ){
			$sp = " ";
		}

		$handle = popen( $smpath . $sp . $smopt, "w" );
		if( !$handle ){
			throw new RuntimeException; 
		}

		try{
			if( $mf === null ){
				$mf = new YggMailFormatter;
			}
			$mf->start( $handle, $mode );
		}
		catch( Exception $e ){
			pclose( $handle );
			throw $e;
		}

		$smobj = new YggSendMail;
		$smobj->_handle = $handle;
		$smobj->_mf     = $mf;

		return $smobj;
	}


	/**
	 * Close sendmail pipe
	 */
	public function close()
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->end();

		$rts = pclose( $this->_handle );
		if( $rts != 0 ){
			throw new RuntimeException;
		}

		$this->_mf     = null;
		$this->_handle = null;
	}


	/**
	 * Send boundary ID
	 *
	 * This method writes boundary ID to stream.<br />
	 * This method can call multipart mode and
	 * current section is body part or header of multi part only.
	 */
	public function boundary()
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		return $this->_mf->boundary();
	}


	/**
	 * Send from header
	 *
	 * This method writes from header.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param string $addr Mail address
	 * @param string $name Name (option)
	 */
	public function from( $addr, $name = "" )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->from( $addr, $name );
	}


	/**
	 * Send "TO" address
	 *
	 * This method writes "TO" address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : name (option)<br />
	 * mail => string : mail address
	 */
	public function toList( $data )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->toList( $data );
	}


	/**
	 * Send "CC" address
	 *
	 * This method writes "CC" address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : Name (option)<br />
	 * mail => string : Mail address
	 */
	public function ccList( $data )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->ccList( $data );
	}


	/**
	 * Send "BCC" address
	 *
	 * This method writes "BCC" address in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param array $data Target data<br />
	 * (It has array in array that has the following elements)<br />
	 * name => string : Name (option)<br />
	 * mail => string : Mail address
	 */
	public function bccList( $data )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->bccList( $data );
	}


	/**
	 * Send subject
	 *
	 * This method writes subject in header section.<br />
	 * This method can be called when current section is top part of header.
	 *
	 * @param string $subject
	 */
	public function subject( $subject )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->subject( $subject );
	}


	/**
	 * Send message
	 *
	 * This method writes message in header section.<br />
	 * This method can be called when current section is header part.
	 *
	 * @param string $msg
	 */
	public function message( $msg )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->message( $msg );
	}


	/**
	 * Send attach data
	 *
	 * This method writes attach data whose file name is $fname to stream.<br />
	 * This method can be called when send mail mode is multipart.
	 *
	 * @param string $fname File name
	 * @param string $data Attach data
	 */
	public function attachData( $fname, $data )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->attachData( $fname, $data );
	}


	/**
	 * Send attach data from file stream
	 *
	 * This method writes data that is specified by $handle that is file stream.<br />
	 * The stream data is named as $fname.<br />
	 * $rrc is specified number of line to read once from stream.
	 * One line length is 72 bytes.
	 *
	 * @param string $fname File name
	 * @param resource $handle File stream
	 * @param integer $rrc Read line count once
	 */
	public function attachFile( $fname, $handle, $rrc = 10 )
	{
		if( $this->_mf === null ){
			throw new BadMethodCallException;
		}

		$this->_mf->attachFile( $fname, $handle, $rrc );
	}
}
?>
