<?php
/**
 * Session class
 *
 * This file defines YggSessionBase class.
 * It includes methods to operate session.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggDore library configuration file
 */
require_once( "YggDore/Base/Config.php" );
/**
 * Require YggWebRequest class
 */
require_once( "YggDore/Base/YggWebRequest.php" );
/**
 * Require YggSessionID class
 */
require_once( "YggDore/Base/YggSessionID.php" );
/**
 * Require YggSessionNotFoundException class
 */
require_once( "YggDore/Base/YggSessionNotFoundException.php" );
/**
 * Require YggSessionDuplicateException class
 */
require_once( "YggDore/Base/YggSessionDuplicateException.php" );
/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );
/**
 * Require YggMobileJP class
 */
require_once( "YggDore/Base/Locale/JP/YggMobileJP.php" );



/**
 * YggDore session class
 *
 * This class supports session operation.
 *
 * <code>
 * $sess = new YggSessionBase;
 * $sess->start();
 * </code>
 *
 * @package YggDore.Base
 */
abstract class YggSessionBase {
	/**
	 * Session configuration
	 */
	protected $_conf;
	/**
	 * Session ID control object
	 */
	private $_sid;
	/**
	 * Session data
	 */
	private $_data;
	/**
	 * Previous access time
	 */
	private $_patime;
	/**
	 * Access time
	 */
	private $_atime;
	/**
	 * Creation time
	 */
	private $_ctime;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 *
	 * @param array $conf Configuration ( Same as YggSessionBase::setConfig )
	 */
	public function __construct( $conf = null )
	{
		$this->_clearData();
		$this->setConfig( $conf );
		$this->dgCreate = new YggDelegate;
		$this->dgLoad   = new YggDelegate;
		$this->dgUpAT   = new YggDelegate;
		$this->dgSave   = new YggDelegate;
		$this->dgDelete = new YggDelegate;
	}


	/**
	 * Get default configuration
	 *
	 * This method returns default configuration data.
	 *
	 * @return array Configuration data<br />
	 * (Same as $conf of YggSessionBase::setConfig method)
	 */
	public static function getDefaultConfig()
	{
		return array(
			"snm" => "session",
			"snc" => 5,
			"dat" => 3600,
			"dct" => 86400,
			"ck"  => array(
				"iskp"   => false,
				"path"   => "/",
				"domain" => "",
				"secure" => false
			)
		);
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_sid === null ){
			return false;
		}
		return true;
	}


	/**
	 * Clear data
	 *
	 * Member variables of the instance is initialized.
	 * This method doesn't delete configuration.
	 */
	protected function _clearData()
	{
		$this->_sid    = null;
		$this->_data   = null;
		$this->_patime = null;
		$this->_atime  = null;
		$this->_ctime  = null;
	}


	/**
	 * Get configuration
	 *
	 * This method returns value of the instance.
	 *
	 * @return array Configuration data<br />
	 * (Same as $conf of YggSessionBase::setConfig method)
	 */
	public function getConfig()
	{
		return $this->_conf;
	}


	/**
	 * Get configuration
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return YggSessionID
	 */
	public function getID()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_sid;
	}


	/**
	 * Get session data area
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return mixed Reference of data area
	 */
	public function &getDataArea()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_data;
	}


	/**
	 * Get previous access time
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return YggDateTime Previous access time
	 */
	public function getPreATime()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_patime;
	}


	/**
	 * Get access time
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return YggDateTime Access time
	 */
	public function getATime()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_atime;
	}


	/**
	 * Get session creation time
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return YggDateTime Creation time
	 */
	public function getCTime()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}
		return $this->_ctime;
	}


	/**
	 * Set configuration
	 *
	 * This method sets configuration to the instance.<br />
	 * If you use this method, the instance data is cleared.
	 * 
	 * @param array $conf Configuration<br />
	 * (It has the following elements)<br />
	 * snm : (string) Session name<br />
	 * snc : (integer) Count of ready session ID when creates new session<br />
	 * dat : (integer) Number of second to disable session from last access time<br />
	 * dct : (integer) Number of second to disable session from session first creation<br />
	 * ck => iskp : (boolean) Keep session cookie mode<br />
	 * ck => path : (string) Path of cookie<br />
	 * ck => domain : (string) Domain of cookie<br />
	 * ck => secure : (boolean) SSL mode of cookie
	 */
	public function setConfig( $conf = null )
	{
		if( $conf === null ){
			$conf = YggTool::callVirtualStatic( $this, "getDefaultConfig" );
		}

		$this->_clearData();
		$this->_conf = $conf;
	}


	/**
	 * Start session processing
	 *
	 * This method starts session processing.<br />
	 * YggSessionBase::save, YggSessionBase::stop or YggSessionBase::delete
	 * method must be call after this method calling.
	 */
	public function start()
	{
		$now = YggDateTime::createNow();
		$ntu = $now->getUnixTimeStamp();

		$sid = new YggSessionID;

		try{
			$sid->set(
				$this->_conf['snm'],
				array(
					YggSessionID::COOKIE,
					YggSessionID::POST,
					YggSessionID::GET
				)
			);
		}
		catch( Exception $e ){
			$this->restart();
			return;
		}

		try{
			$lsd = $this->_loadData( $sid );
		}
		catch( YggSessionNotFoundException $e ){
			$this->restart();
			return;
		}
			
		$ctu = $lsd['ct']->getUnixTimeStamp();
		if( $this->_conf['dct'] >= 0 && $ntu - $ctu > $this->_conf['dct'] ){
			$this->_deleteData( $sid );
			$this->restart();
			return;
		}
		$atu = $lsd['at']->getUnixTimeStamp();
		if( $this->_conf['dat'] >= 0 && $ntu - $atu > $this->_conf['dat'] ){
			$this->_deleteData( $sid );
			$this->restart();
			return;
		}

		$this->_updateATimeData( $sid, $now );

		$this->_sid    = $sid;
		$this->_data   = $lsd['data'];
		$this->_patime = $lsd['at'];
		$this->_atime  = $now;
		$this->_ctime  = $lsd['ct'];
	}

		
	/**
	 * Restart session processing
	 *
	 * This method starts session processing.<br />
	 * This method recreates new session.<br />
	 * YggSessionBase::save, YggSessionBase::stop or YggSessionBase::delete
	 * method must be call after this method calling.
	 */
	public function restart()
	{
		$now = YggDateTime::createNow();

		$nsid = $this->_createSession( $now );
		$this->_sendCookie( $nsid );
		
		$this->_sid    = $nsid;
		$this->_data   = null;
		$this->_patime = null;
		$this->_atime  = $now;
		$this->_ctime  = $now;
	}


	/*
	 * Create a new session
	 *
	 * This method is called when YggSessionBase::start is called.<br />
	 * Session data is initialized from arguments.
	 *
	 * @param YggSessionID $sid Session ID
	 * @param YggDateTime $ct Creation time
	 */
	protected abstract function _createData( $sid, $ct );


	/*
	 * Load session
	 *
	 * This method is called when YggSessionBase::start is called.<br />
	 * The session data matched with $sid is loaded.
	 *
	 * @param YggSessionID $sid Session ID
	 * @return array Session data
	 */
	protected abstract function _loadData( $sid );


	/*
	 * Update last access time
	 *
	 * This method is called when YggSessionBase::start is called.<br />
	 * Last access time matched with $sid record is updated to current time.
	 *
	 * @param YggSessionID $sid Session ID
	 * @param YggDateTime $now Current time
	 */
	protected abstract function _updateATimeData( $sid, $now );


	/*
	 * Save session
	 *
	 * This method is called when calls
	 * YggSessionBase::save or YggSessionBase::stop methods.<br />
	 * session data specified by $data is saved.
	 *
	 * @param string $sid Session ID
	 * @param mixed $data Session data
	 */
	protected abstract function _saveData( $sid, $data );


	/*
	 * Delete session
	 *
	 * This method is called when calls YggSessionBase::delete methods.<br />
	 * session data specified by $sid is deleted.
	 *
	 * @param string $sid Session ID
	 */
	protected abstract function _deleteData( $sid );


	/**
	 * Send cookie
	 *
	 * @param YggSessionID $sid Session ID
	 */
	private function _sendCookie( $sid )
	{
		$ltime = 0;
		if( $this->_conf['ck']['iskp'] ){
			$ltime = YGG_INTMAX;
		}

		$sid->sendCookie(
			$ltime,
			$this->_conf['ck']['path'],
			$this->_conf['ck']['domain'],
			$this->_conf['ck']['secure']
		);
	}


	/**
	 * Send cookie for deleting
	 */
	private function _sendDeleteCookie()
	{
		YggSessionID::sendDeleteCookie(
			$this->_conf['snm'],
			$this->_conf['ck']['path'],
			$this->_conf['ck']['domain'],
			$this->_conf['ck']['secure']
		);
	}


	/**
	 * Send request to create a session to data object
	 *
	 * This method throws request and data to delegate that stores a session ID.
	 * Usable session ID is returned.
	 *
	 * @param YggDateTime $ct Creation time
	 *
	 * @return YggSessionID Session ID
	 */
	private function _createSession( $ct )
	{
		$nsid = new YggSessionID;
		for( $i = 0; $i < $this->_conf['snc']; $i++ ){
			$nsid->set( $this->_conf['snm'], array( YggSessionID::CREATE ) );

			try{
				$this->_createData( $nsid, $ct );
			}
			catch( YggSessionDuplicateException $e ){
				continue;
			}

			break;
		}
		if( $i >= $this->_conf['snc'] ){
			throw new YggSessionDuplicateException;
		}

		return $nsid;
	}


	/**
	 * Send request to save session data
	 *
	 * This method throws request and data to delegate that stores session data.
	 *
	 * @param mixed $param Data to transfer to each delegate
	 */
	public function save()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$this->_saveData( $this->_sid, $this->_data );
	}


	/**
	 * Stop session
	 *
	 * This method calls delegate of saving session data and
	 * member variables is initialized in the instance.
	 */
	public function stop()
	{
		$this->save();
		$this->_clearData();
	}


	/**
	 * Delete session
	 *
	 * This method deletes cookie and calls session deletion delegate.
	 */
	public function delete()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$this->_deleteData( $this->_sid );
		$this->_sendDeleteCookie();
		$this->_clearData();
	}
}
?>
