<?php
/**
 * Session ID class
 *
 * This file defines YggSessionID class.
 * It includes method to manage session ID.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );
/**
 * Require YggHTML class
 */
require_once( "YggDore/Base/YggHTML.php" );
/**
 * Require YggWebRequest class
 */
require_once( "YggDore/Base/YggWebRequest.php" );


/**
 * Session ID class
 *
 * This class has methods to manage session ID.
 *
 * <code>
 * $sidobj = new YggSessionID;
 * $sidobj->set(
 *     "name",
 *     array(
 *         YggSessionID::COOKIE,
 *         YggSessionID::REQUEST,
 *         YggSessionID::CREATE
 *     )
 * );
 * $sidobj->clear();
 * </code>
 *
 * @package YggDore.Base
 */
class YggSessionID {
	/**
	 * Mode of getting session ID from $_COOKIE
	 */
	const COOKIE  = 1;
	/**
	 * Mode of getting session ID from $_REQUEST
	 */
	const REQUEST = 2;
	/**
	 * Mode of getting session ID from $_GET
	 */
	const GET     = 3;
	/**
	 * Mode of getting session ID from $_POST
	 */
	const POST    = 4;
	/**
	 * Mode of creating new session ID
	 */
	const CREATE  = 5;


	/**
	 * Session name
	 */
	private $_name;
	/**
	 * Session ID
	 */
	private $_id;
	/**
	 * The way of session ID getting
	 */
	private $_order;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->clear();
	}


	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_name  = null;
		$this->_id    = null;
		$this->_order = null;
	}


	/**
	 * Get session name
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return string
	 */
	public function getName()
	{
		if( $this->_name === null ){
			throw new BadMethodCallException;
		}
		return $this->_name;
	}


	/**
	 * Get session ID
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return string
	 */
	public function getID()
	{
		if( $this->_id === null ){
			throw new BadMethodCallException;
		}
		return $this->_id;
	}


	/**
	 * Get way of session ID getting
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer Way of getting session ID<br />
	 * YggSessionID::COOKIE : From $_COOKIE<br />
	 * YggSessionID::REQUEST : From $_REQUEST<br />
	 * YggSessionID::GET : From $_GET<br />
	 * YggSessionID::POST : From $_POST<br />
	 * YggSessionID::CREATE : Creating new
	 */
	public function getOrder()
	{
		if( $this->_order === null ){
			throw new BadMethodCallException;
		}
		return $this->_order;
	}


	/**
	 * Set session information
	 *
	 * This method searches and gets session ID.<br />
	 * Searching order is specified by $oltbl. It can set multi orders.<br />
	 * $name is cookie name or web request key name.
	 *
	 * @param string $name Session name
	 * @param array $oltbl Search order list<br />
	 * (It is able to be selected options as follows)<br />
	 * YggSessionID::COOKIE : From $_COOKIE<br />
	 * YggSessionID::REQUEST : From $_REQUEST<br />
	 * YggSessionID::GET : From $_GET<br />
	 * YggSessionID::POST : From $_POST<br />
	 * YggSessionID::CREATE : Creating new session ID
	 */
	public function set( $name, $oltbl )
	{
		if( $name == "" ){
			throw new UnexpectedValueException;
		}
		elseif( strlen($name) > 32 ||
		        !preg_match( '/^[0-9a-zA-Z]+$/', $name ) ){
			throw new UnexpectedValueException;
		}
		$olcnt = count( $oltbl );
		if( $olcnt <= 0 ){
			throw new UnexpectedValueException;
		}
		for( $i = 0; $i < $olcnt; $i++ ){
			if( $oltbl[$i] < self::COOKIE || $oltbl[$i] > self::CREATE ){
				throw new UnexpectedValueException;
			}
		}

		$datatbl = array(
			"getCookie",
			"getRequest",
			"getGet",
			"getPost"
		);
		$id    = "";
		$order = null;
		for( $i = 0; $i < $olcnt; $i++ ){
			if( $oltbl[$i] == self::CREATE ){
				$id    = YggTool::uniqueID();
				$order = $oltbl[$i];
				break;
			}
			
			$id = call_user_func(
				array( "YggWebRequest", $datatbl[$oltbl[$i]-1] ),
				$name,
				32
			);
			if( $id == "" ){
				continue;
			}
			elseif( strlen($id) > 32 ||
			        !preg_match( '/^[0-9a-zA-Z]+$/', $id ) ){
				throw new UnexpectedValueException;
			}

			$order = $oltbl[$i];
			break;
		}
		if( $order === null ){
			throw new UnexpectedValueException;
		}

		$this->_id    = $id;
		$this->_name  = $name;
		$this->_order = $order;
	}


	/**
	 * Send session ID cookie
	 *
	 * This method sends session ID cookie in the instance to the client.
	 *
	 * @param integer $ltime Life second time of cookie
	 * @param string $path Path of cookie
	 * @param string $domain Domain of cookie
	 * @param boolean $secure SSL mode of cookie
	 */
	public function sendCookie( $ltime = 0, $path = "/", $domain = "", $secure = false )
	{
		$name = $this->getName();
		$id   = $this->getID();

		if( $ltime < 0 ){
			throw new UnexpectedValueException;
		}

		$rts = setcookie(
			$name, $id, $ltime, $path, $domain, $secure
		);
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/**
	 * Create session data fomatted as url encode
	 *
	 * This method returns session data that is converted url encode format.<br />
	 * Empty string is returned when session data is got from cookie. This is same as PHP's SID.
	 *
	 * @return string
	 */
	public function getKeepURL()
	{
		$order = $this->getOrder();
		if( $order == self::COOKIE ){
			return "";
		}

		$name = urlencode( $this->getName() );
		if( $name == "" ){
			throw new RuntimeException;
		}
		$id = urlencode( $this->getID() );
		if( $id == "" ){
			throw new RuntimeException;
		}

		return $name . "=" . $id;
	}


	/**
	 * Create session data fomatted as url encode ( Amparsand is inserted to the first )
	 *
	 * This method returns session data that is converted url encode format.<br />
	 * The format is [&name=id].<br />
	 * Empty string is returned when session data is got from cookie. This is same as PHP's SID.
	 *
	 * @return string
	 */
	public function getKeepURLAmp()
	{
		$rts = $this->getKeepURL();
		if( $rts != "" ){
			$rts = "&" . $rts;
		}

		return $rts;
	}


	/**
	 * Create session data fomatted as url encode ( Question is inserted to the first )
	 *
	 * This method returns session data that is converted url encode format.<br />
	 * The format is [?name=id].<br />
	 * Empty string is returned when session data is got from cookie. This is same as PHP's SID.
	 *
	 * @return string
	 */
	public function getKeepURLQuot()
	{
		$rts = $this->getKeepURL();
		if( $rts != "" ){
			$rts = "?" . $rts;
		}

		return $rts;
	}


	/**
	 * Create session data fomatted as html(xhtml) hidden tag
	 *
	 * This method returns session data that is converted html(xhtml) hidden tag format.<br />
	 * The format is [<input type="hidden" name="name" value="value">].<br />
	 * If $isx is specified TRUE, hidden tag formatted as XHTML is returned.<br />
	 * Empty string is returned when session data is got from cookie. This is same as PHP's SID.
	 *
	 * @param boolean $isx XHTML mode
	 * @return string
	 */
	public function getKeepHiddenTag( $isx = true )
	{
		$order = $this->getOrder();
		if( $order == self::COOKIE ){
			return "";
		}

		$xtag = "";
		if( $isx ){
			$xtag = " /";
		}

		$name = YggHTML::specialChars( $this->getName() );
		if( $name == "" ){
			throw new RuntimeException;
		}
		$id = YggHTML::specialChars( $this->getID() );
		if( $id == "" ){
			throw new RuntimeException;
		}

		return '<input type="hidden" name="' . $name . '" value="' .
		       $id . '"' . $xtag . '>';
	}


	/**
	 * Create URL encode
	 *
	 * This method creates URL encode from $data.<br />
	 * The difference with YggHTML class is return value of this method includes session ID.<br />
	 * $exttbl is target key name list in $data.<br />
	 * Default setting is that key name of URL encode is used key name of each $data elements.
	 * Key name of URL encode can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $exttbl Target key list
	 * @param array $kntbl Key name list
	 * @return string
	 * @see YggHTML::createURLEncode()
	 */
	public function createURLEncode(
		$data, $exttbl, $kntbl = null
	)
	{
		$kpurl = $this->getKeepSessionIDURL();

		$dturl = YggHTML::createURLEncode( $data, $exttbl, $kntbl );

		$asp = "";
		if( $kpurl != "" && $dturl != "" ){
			$asp = "&";
		}

		return $dturl . $asp . $kpurl;
	}


	/**
	 * Create URL encode from all elements in array
	 *
	 * This method creates URL encode from all elements in $data.<br />
	 * The difference with YggHTML class is return value of this method includes session ID.<br />
	 * $exttbl is target key name list in $data.<br />
	 * Default setting is that key name of URL encode is used key name of each $data elements.
	 * Key name of URL encode can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $kntbl Key name list
	 * @return string
	 * @see YggHTML::createAllURLEncode()
	 */
	public function createAllURLEncode( $data, $kntbl = null )
	{
		if( count( $data ) <= 0 ){
			return "";
		}

		return $this->createURLEncode( $data, array_keys( $data ), $kntbl );
	}


	/**
	 * Create HTML(XHTML) hidden tag
	 *
	 * This method creates HTML(XHTML) hidden tag from $data.<br />
	 * The difference with YggHTML class is return value of this method includes session ID.<br />
	 * $exttbl is target key name list in $data.<br />
	 * Default setting is that name of tag is used key name of each $data elements.
	 * Hidden tag name can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * If $isx is specified TRUE, XHTML tag creates.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $exttbl Target key list
	 * @param array $kntbl Tag name list
	 * @param boolean $isx XHTML mode
	 * @return string
	 * @see YggHTML::createHiddenTag()
	 */
	public function createHiddenTag( $data, $exttbl, $kntbl = null, $isx = true )
	{
		$kptag = $this->getKeepHiddenTag( $isx );

		$dttag = YggHTML::createHiddenTag( $data, $exttbl, $kntbl, $isx );

		return $dttag . $kptag;
	}


	/**
	 * Create HTML(XHTML) hidden tags from all elements in array
	 *
	 * This method creates HTML(XHTML) hidden tag from all elements in $data.<br />
	 * Default setting is that name of tag is used key name of each $data elements.
	 * Hidden tag name can be changed when
	 * $kntbl is specified pair elements whose
	 * key is $data key and value is URL encode key name.<br />
	 * If $isx is specified TRUE, XHTML tag creates.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param array $data Target data
	 * @param array $kntbl Tag name list
	 * @param boolean $isx XHTML mode
	 * @return string
	 * @see YggHTML::createAllHiddenTag()
	 */
	public function createAllHiddenTag( $data, $kntbl = null, $isx = true )
	{
		if( count( $data ) <= 0 ){
			return "";
		}

		return $this->createHiddenTag( $data, array_keys( $data ), $kntbl, $isx );
	}
	

	/**
	 * Send deleting cookie
	 *
	 * This method sends deletion session ID cookie in the instance to the client.
	 *
	 * @param string $name Session name
	 * @param string $path Path of cookie
	 * @param string $domain Domain of cookie
	 * @param boolean $secure SSL mode of cookie
	 */
	public static function sendDeleteCookie( $name, $path = "/", $domain = "", $secure = false )
	{
		if( $name == "" ){
			throw new UnexpectedValueException;
		}
		elseif( strlen($name) > 32 ||
		        !preg_match( '/^[0-9a-zA-Z]+$/', $name ) ){
			throw new UnexpectedValueException;
		}

		$rts = setcookie(
			$name, "", time() - 3600, $path, $domain, $secure
		);
		if( !$rts ){
			throw new RuntimeException;
		}
	}
}
?>
