<?php
/**
 * Session handler class for mysql
 *
 * This file defines YggSessionMySQLi class.
 * It includes methods to store session data to mysql database.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggMySQLi class
 */
require_once( "YggDore/Base/YggMySQLi.php" );
/**
 * Require YggSessionBase class
 */
require_once( "YggDore/Base/YggSessionBase.php" );
/**
 * Require YggDateTime class
 */
require_once( "YggDore/Base/YggDateTime.php" );



/**
 * Query to insert new session data
 */
define(
	"YSMI_QUERY_CREATE",
	"INSERT INTO " .
		"%tblnm% " .
	"VALUES " .
		"( %skey%, NULL, %ct%, %ct%, 1 )"
);

/**
 * Query to load session data
 */
define(
	"YSMI_QUERY_LOAD",
	"SELECT " .
		"DATA, " .
		"DATE_FORMAT( ATIME, '%%Y/%%m/%%d %%H:%%i:%%s' ) AS AT, " .
		"DATE_FORMAT( CTIME, '%%Y/%%m/%%d %%H:%%i:%%s' ) AS CT " .
	"FROM " .
		"%tblnm% " .
	"WHERE " .
		"SKEY = %skey% AND " .
		"ISACT = 1"
);

/**
 * Query to update last access time
 */
define(
	"YSMI_QUERY_UPDAT",
	"UPDATE " .
		"%tblnm% " .
	"SET " .
		"ATIME = %nowt% " .
	"WHERE " .
		"SKEY = %skey%"
);

/**
 * Query to save session data
 */
define(
	"YSMI_QUERY_SAVE",
	"UPDATE " .
		"%tblnm% " .
	"SET " .
		"DATA = %data% " .
	"WHERE " .
		"SKEY = %skey%"
);

/**
 * Query to disable session ID
 */
define(
	"YSMI_QUERY_DELETE",
	"UPDATE " .
		"%tblnm% " .
	"SET " .
		"DATA = NULL, " .
		"ISACT = 0 " .
	"WHERE " .
		"SKEY = %skey%"
);

/**
 * Query to clean session data in database
 */
define(
	"YSMI_QUERY_CLEAN",
	"DELETE FROM " .
		"%tblnm% " .
	"WHERE " .
		"CTIME <= NOW() - INTERVAL %kprs% SECOND"
);



/**
 * Session handler class for mysql
 *
 * This class has methods to store session data to mysql database.
 *
 * <code>
 * $mysqli  = new YggMySQLi( "localhost", "hoge", "pass", "dbname" );
 * $sess = new YggSessionMySQLi( $mysqli );
 * $sess->start();
 * </code>
 *
 * This class uses the following database table.
 * 
 * <code>
 * CREATE TABLE SESSION (
 *     SKEY  VARCHAR(32) BINARY NOT NULL,          -- Session key
 *     DATA  TEXT,                                 -- Data
 *     CTIME DATETIME NOT NULL,                    -- Creation time
 *     ATIME DATETIME NOT NULL,                    -- Last access time
 *     ISACT BOOLEAN NOT NULL,                     -- Active flag
 *     PRIMARY KEY ( SKEY ),
 *     KEY INDEX_CTIME ( CTIME ),
 *     KEY INDEX_ATIME ( ATIME ),
 *     KEY INDEX_ISACT ( ISACT )
 * );
 * </code>
 *
 * @package YggDore.Base
 */
class YggSessionMySQLi extends YggSessionBase {
	/**
	 * Connection resource
	 */
	protected $_mysqli;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 *
	 * 
	 * @param YggMySQLi $mysqli Database connection
	 * @param array $conf Configuration<br />
	 * It has YggSessionBase's configuration elements and the following elements.<br />
	 * tbn : (string) Table name
	 */
	public function __construct( $mysqli, $conf = null )
	{
		parent::__construct( $conf );
		$this->setMySQLi( $mysqli );
	}


	/**
	 * Get default configuration
	 *
	 * This method returns default configuration data.
	 *
	 * @return array Configuration data<br />
	 * (Same as $conf of YggSessionMySQLi::__construct method)
	 */
	public static function getDefaultConfig()
	{
		$conf = parent::getDefaultConfig();
		$conf['tbn'] = "SESSION";

		return $conf;
	}


	/**
	 * Get mysqli connection
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return YggMySQLi
	 */
	public function getMySQLi()
	{
		return $this->_mysqli;
	}


	/**
	 * Set mysqli connection
	 *
	 * This method sets mysqli connection to the instance.<br />
	 * If you use this method, the instance data is cleared.
	 * 
	 * @param YggMySQLi $mysqli Database connection
	 */
	public function setMySQLi( $mysqli )
	{
		$this->_mysqli = $mysqli;
		$this->_clearData();
	}


	/*
	 * Clean
	 *
	 * This method is deleted records in database matched with the arguments.<br />
	 * $kprs is time to keep record after it expires.
	 *
	 * @param integer $kprs Number of seconds to delete record after creation record time
	 */
	public function cleanData( $kprs )
	{
		if( $kprs < 0 ){
			throw new RangeException;
		}

		$sql = $this->_mysqli->sql(
			YSMI_QUERY_CLEAN,
			array(
				"tblnm" => $this->_conf['tbn'],
				"kprs"  => $kprs
			),
			array(
				"tblnm" => YggMySQLi::FT,
				"kprs"  => YggMySQLi::INT
			)
		);
		$rts = $this->_mysqli->query( $sql );
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/*
	 * Create a new session
	 *
	 * This method is called when YggSessionBase::start is called.<br />
	 * Session data is initialized from arguments.
	 *
	 * @param YggSessionID $sid Session ID
	 * @param YggDateTime $ct Creation time
	 */
	protected function _createData( $sid, $ct )
	{
		$sql = $this->_mysqli->sql(
			YSMI_QUERY_CREATE,
			array(
				"tblnm" => $this->_conf['tbn'],
				"skey"  => $sid->getID(),
				"ct"    => $ct->getString( "Y/m/d H:i:s" )
			),
			array(
				"tblnm" => YggMySQLi::FT,
				"skey"  => YggMySQLi::STRING,
				"ct"    => YggMySQLi::STRING
			)
		);
		$rts = $this->_mysqli->query( $sql );
		if( !$rts ){
			throw new RuntimeException;
		}
		elseif( $this->_mysqli->affected_rows <= 0 ){
			throw new YggSessionDuplicateException;
		}
	}


	/*
	 * Load session
	 *
	 * This method is called when YggSessionBase::start is called.<br />
	 * The session data matched with $sid is loaded.
	 *
	 * @param YggSessionID $sid Session ID
	 * @return array Session data
	 */
	protected function _loadData( $sid )
	{
		$sql = $this->_mysqli->sql(
			YSMI_QUERY_LOAD,
			array(
				"tblnm" => $this->_conf['tbn'],
				"skey"  => $sid->getID()
			),
			array(
				"tblnm" => YggMySQLi::FT,
				"skey"  => YggMySQLi::STRING
			)
		);
		$result = $this->_mysqli->query( $sql );
		if( !$result ){
			throw new RuntimeException;
		}
		elseif( $result->num_rows <= 0 ){
			$result->free();
			throw new YggSessionNotFoundException;
		}
		$row = $result->fetch_assoc();
		$result->free();

		$ct = new YggDateTime;
		$ct->setString( $row['CT'] );
		$at = new YggDateTime;
		$at->setString( $row['AT'] );

		$rts = array(
			"data"  => null,
			"ct"    => $ct,
			"at"    => $at
		);
		if( $row['DATA'] !== null ){
			$rts['data'] = unserialize( $row['DATA'] );
		}

		return $rts;
	}


	/*
	 * Update last access time
	 *
	 * This method is called when YggSessionBase::start is called.<br />
	 * Last access time matched with $sid record is updated to current time.
	 *
	 * @param YggSessionID $sid Session ID
	 * @param YggDateTime $now Current time
	 */
	protected function _updateATimeData( $sid, $now )
	{
		$sql = $this->_mysqli->sql(
			YSMI_QUERY_UPDAT,
			array(
				"tblnm" => $this->_conf['tbn'],
				"skey"  => $sid->getID(),
				"nowt"  => $now->getString( "Y/m/d H:i:s" )
			),
			array(
				"tblnm" => YggMySQLi::FT,
				"skey"  => YggMySQLi::STRING,
				"nowt"  => YggMySQLi::STRING
			)
		);
		$rts = $this->_mysqli->query( $sql );
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/*
	 * Save session
	 *
	 * This method is called when calls
	 * YggSessionBase::save or YggSessionBase::stop methods.<br />
	 * session data specified by $data is saved.
	 *
	 * @param string $sid Session ID
	 * @param mixed $data Session data
	 */
	protected function _saveData( $sid, $data )
	{
		$sql = $this->_mysqli->sql(
			YSMI_QUERY_SAVE,
			array(
				"tblnm" => $this->_conf['tbn'],
				"skey"  => $sid->getID(),
				"data"  => serialize( $data )
			),
			array(
				"tblnm" => YggMySQLi::FT,
				"skey"  => YggMySQLi::STRING,
				"data"  => YggMySQLi::STRING
			)
		);
		$rts = $this->_mysqli->query( $sql );
		if( !$rts ){
			throw new RuntimeException;
		}
	}


	/*
	 * Delete session
	 *
	 * This method is called when calls YggSessionBase::delete methods.<br />
	 * session data specified by $sid is deleted.
	 *
	 * @param string $sid Session ID
	 */
	protected function _deleteData( $sid )
	{
		$sql = $this->_mysqli->sql(
			YSMI_QUERY_DELETE,
			array(
				"tblnm" => $this->_conf['tbn'],
				"skey"  => $sid->getID()
			),
			array(
				"tblnm" => YggMySQLi::FT,
				"skey"  => YggMySQLi::STRING
			)
		);
		$rts = $this->_mysqli->query( $sql );
		if( !$rts ){
			throw new RuntimeException;
		}
	}
}
?>
