<?php
/**
 * String class
 *
 * This file defines YggString class.
 * It includes methods to adjust return code and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require config file
 */
require_once( "YggDore/Base/Config.php" );
/**
 * Require YggSafe class
 */
require_once( "YggDore/Base/YggSafe.php" );
/**
 * Require YggArray class
 */
require_once( "YggDore/Base/YggArray.php" );
/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );


/**
 * String class 
 *
 * This class has methods to adjust return code and so on.
 *
 * @package YggDore.Base
 */
class YggString {
	/**
	 * Left trim mode
	 */
	const LTRIM = 0x01;
	/**
	 * Right trim mode
	 */
	const RTRIM = 0x02;
	/**
	 * Both trim mode
	 */
	const TRIM  = 0x03;

	/**
	 * Converting to system end of line mode
	 */
	const EOL  = 1;
	/**
	 * Converting to CR end of line mode
	 */
	const CR   = 2;
	/**
	 * Converting to LF end of line mode
	 */
	const LF   = 3;
	/**
	 * Converting to CRLF end of line mode
	 */
	const CRLF = 4;


	/**
	 * Replace from between % character to word
	 *
	 * This method searches point between % character in $src
	 * and replaces from %name% to $chtbl[name] in it.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $src = "%name1% %name2%";
	 * $chtbl = array(
	 *     "name1" => "fujiyama",
	 *     "name2" => "geisya"
	 * );
	 * $dst = YggString::embed($src, $chtbl);
	 * (Result)
	 * $dst => fujiyama geisya
	 * </code>
	 *
	 * @param string $src Target string
	 * @param array $chtbl Replace list<br />
	 * Its key is used to look up the word in $data and replace to value.
	 * @return string
	 */
	public static function embed( $src, $chtbl )
	{
		$chgtbl = array( "%%" => "%" );

		if( count($chtbl) <= 0 ){
			throw new UnexpectedValueException;
		}

		if( $src == "" ){
			return "";
		}

		foreach( $chtbl as $key => $val ){
			if( strlen($key) <= 0 ){
				throw new UnexpectedValueException;
			}

			$chgtbl['%'.$key.'%'] = $val;
		}
		return (string)preg_replace(
			'/(%[^%]*%)/e',
			'isset($chgtbl[\'$1\']) ? $chgtbl[\'$1\'] : ""',
			$src
		);
	}


	/**
	 * Trim space
	 * 
	 * This method trims left or/and right space code.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $data = "  Hello1!Good1!  ";
	 * $data = YggString::trim( $data, YggString::TRIM );
	 * (Result)
	 * $data => "Hello1!Good1!"
	 * </code>
	 *
	 * @param string $data Target string
	 * @param integer $adjf Trim mode<br />
	 * YggString::TRIM : Both trimming mode<br />
	 * YggString::LTRIM : Left trimming mode<br />
	 * YggString::RTRIM : Right trimming mode
	 * @return string
	 */
	static public function trim( $data, $adjf = self::TRIM )
	{
		if( !($adjf & 0x03) ){
			throw new UnexpectedValueException;
		}

		if( $data == "" ){
			return "";
		}

		if( $adjf & self::RTRIM ){
			$data = preg_replace( '/[[:space:]]+$/', "", $data );
		}
		if( $adjf & self::LTRIM ){
			$data = preg_replace( '/^[[:space:]]+/', "", $data );
		} 

		return (string)$data;
	}


	/**
	 * Optimize end of line code
	 * 
	 * This method replaces from CR, LF or CRLF code to eol code in $data.<br />
	 * EOL is CR, LF, CRLF or YGG_EOL. YGG_EOL is specified in Config.php.<br />
	 * $maxret is specified maximum number allowed repeating EOL.
	 * If number of repeating EOL is over $maxret, they fix $maxret EOLs. <br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $data = "Hello1!\n\nGood1!\n";
	 * $data = YggString::eol( $data, 0, YggString::LF );
	 * (Result)
	 * $data => "Hello1!Good1!"
	 * </code>
	 *
	 * @param string $data Target string
	 * @param integer $maxret Max Repeating EOL count<br />
	 * @param integer $retchr EOL code<br />
	 * YggString::EOL : YGG_EOL in Config.php<br />
	 * YggString::LF : LF<br />
	 * YggString::CR : CR<br />
	 * YggString::CRLF : CR+LF
	 * @return string
	 */
	public static function eol(
		$data, $maxret = 0, $retchr = self::EOL
	)
	{
		if( $retchr < self::EOL || $retchr > self::CRLF ){
			throw new UnexpectedValueException;
		}

		if( $data == "" ){
			return "";
		}

		$data = preg_replace(
			'/(\r\n|\r|\n)/', YGG_EOL, $data
		);
		if( $data == "" ){
			throw new RuntimeException;
		}

		$rettbl = array( YGG_EOL, "\r", "\n", "\r\n" );
		$rc = $rettbl[$retchr-self::EOL];

		if( $rc != YGG_EOL ){
			$data = str_replace( YGG_EOL, $rc, $data );
			if( $data == "" ){
				throw new RuntimeException;
			}
		}

		if( $maxret < 0 ){
			return $data;
		}
		elseif( $maxret == 0 ){
			return (string)str_replace( $rc, "", $data );
		}

		$ptnrc = str_repeat( $rc, $maxret );
		if( $ptnrc == "" ){
			throw new RuntimeException;
		}

		return (string)preg_replace(
			'/(' . $rc . '){' . $maxret . ',}/',
			$ptnrc,
			$data
		);
	}


	/**
	 * Replace multi spaces
	 * 
	 * This method replaces from multi spaces code to $sp in $data.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $data = "  a   bc  10 ";
	 * $data = YggString::spaceOne( $data );
	 * (Result)
	 * $data => " a bc 10 "
	 * </code>
	 *
	 * @param string $data Target data
	 * @param string $sp String of replacing to
	 * @return string
	 */
	public static function spaceOne( $data, $sp = " " )
	{
		if( $data == "" ){
			return "";
		}

		return (string)preg_replace( '/[ ]+/', $sp, $data );
	}


	/**
	 * Create random characters string
	 * 
	 * This method creates random digit and alphabet, $addstr characters string.
	 *
	 * <code>
	 * $rts = YggString::random( 15, "%-_" );
	 * </code>
	 *
	 * @param integer $len String length
	 * @param string $addstr Random addition string
	 * @return string
	 */
	public static function random( $len, $addstr = "" )
	{
		if( $len <= 0 ){
			throw new RangeException;
		}

		$pstbl = "0123456789" .
		         "abcdefghijkmnopqrstuvwxyz" .
				 "ABCDEFGHJKLMNPQRSTUVWXYZ";

		$pstbl .= $addstr;

		$slen = strlen( $pstbl );
		if( $slen <= 0 ){
			throw new RuntimeException;
		}

		$rts = "";
		for( $i = 0; $i < $len; $i++ ){
			$idx = mt_rand( 0, $slen - 1 );

			$rts .= substr( $pstbl, $idx, 1 );
		}

		return $rts;
	}


	/**
	 * Convert to stable string
	 * 
	 * This method converts type of $data to string.<br />
	 * If string length of $data is more than $limit, it fixes to $limit.
	 * If NULL code is found in it, it is cut before NULL.<br />
	 * This function defends NULL byte attach.<br />
	 * This function doesn't send warning message when type of $data is anything.
	 *
	 * <code>
	 * $data = "150\0test";
	 * $rts  = YggString::stable( $data, 1024 );
	 * (Result)
	 * $rts => "150"
	 * </code>
	 *
	 * @param mixed $data Target string
	 * @param integer $limit Limit byte length
	 * @return string
	 */
	public static function stable( $data, $limit = YGG_YSSLIMIT )
	{
		if( $limit < 0 ){
			throw new UnexpectedValueException;
		}

		$data = YggSafe::safeString( $data );
		if( $data == "" ){
			return "";
		}

		$data = substr( $data, 0, $limit );

		$pos  = strpos( $data, "\0" );
		if( $pos !== false ){
			$data = substr( $data, 0, $pos );
			if( $data == "" ){
				throw new RuntimeException;
			}
		}

		return YggString::eol( $data, -1 );
	}


	/**
	 * Convert to stable strings in array
	 * 
	 * This method converts type of each elements specified keys by $keyl in $data to string.<br />
	 * If string length of $data elements is more than $limit, it fixes to $limit.
	 * If NULL code is found in each elements, the string of the elements is cut before NULL.
	 * This function defends NULL byte attach.<br />
	 * This function doesn't send warning message when type of elements of $data is anything.
	 *
	 * <code>
	 * $data = array( "name" => "Ito\0yan", "test" => 15 );
	 * $rts  = YggString::stableArray(
	 *     $data, array( "name" ), 1024
	 * );
	 * (Result)
	 * $rts => array( "name" => "Ito" )
	 * </code>
	 *
	 * @param array $data Target string array
	 * @param array $keyl Target key list
	 * @param integer $limit Limit byte length
	 * @return array
	 */
	public static function stableArray( $data, $keyl, $limit = YGG_YSSLIMIT )
	{
		$data = YggSafe::safeArray( $data );

		$data = YggArray::extract( $data, $keyl );

		foreach( $keyl as $key ){
			$data[$key] = self::stable( $data[$key], $limit );
		}

		return $data;
	}
}
?>
