<?php
/**
 * Common tool class
 *
 * This file defines YggTool class.
 * It includes methods to create unique ID and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggSafe class
 */
require_once( "YggDore/Base/YggSafe.php" );



/**
 * Common tool class
 *
 * This class has methods to create unique ID and so on.
 *
 * @package YggDore.Base
 */
class YggTool {
	/**
	 * Compare mode past
	 *
	 * This value is used by YggTool::cmpPHPVer method.
	 * It shows past version.
	 */
	const PAST = -1;
	/**
	 * Compare mode same
	 *
	 * This value is used by YggTool::cmpPHPVer method.
	 * It shows same version.
	 */
	const SAME = 0;
	/**
	 * Compare mode future
	 *
	 * This value is used by YggTool::cmpPHPVer method.
	 * It shows future version.
	 */
	const FUTURE = 1;


	/**
	 * Converts value
	 *
	 * This method converts $value to matched value of $ctbl.
	 *
	 * <code>
	 * $ctbl = array(
	 *     "case" => array(
	 *         "Test" => "FUJI",
	 *         "test" => "fuji",
	 *         ""     => "empty"
	 *     )
	 *     "default" => "OTHER"
	 * );
	 * 
	 * $value1 = "Test";
	 * $rts1   = YggTool::convCase( $value, $ctbl );
	 * 
	 * $value2 = "Abc";
	 * $rts2   = YggTool::convCase( $value, $ctbl );
	 *
	 * $value3 = "";
	 * $rts3   = YggTool::convCase( $value, $ctbl );
	 * 
	 * (Result)
	 * $rts1 => "FUJI"
	 * $rts2 => "OTHER"
	 * $rts3 => "empty"
	 * </code>
	 *
	 * @param string $value Target value
	 * @param array $ctbl Convert information array<br />
	 * (It has the following elements)<br />
	 * case => Case list in which the key is case of $value and value is replaced there<br />
	 * default => Default data if $value doesn't matches in $ctbl['case'].
	 * NULL is used instead of it when it is undefined.
	 * @return mixed
	 */
	public static function convCase( $value, $ctbl )
	{
		if( !isset($ctbl['default']) ){
			$ctbl['default'] = null;
		}

		if( !isset( $ctbl['case'][$value] ) ){
			return $ctbl['default'];
		}

		return $ctbl['case'][$value];
	}


	/**
	 * Converts value in array
	 *
	 * This method converts $data to matched value of $ctbl.
	 * This method is array version of YggTool::convCase method.
	 *
	 * <code>
	 * $catbl = array(
	 *     "val1" => array(
	 *         "case" => array(
	 *             "Test" => "Fuji1",
	 *             "test" => "fuji1",
	 *             ""     => "empty1"
	 *         ),
	 *         "default" => "other1"
	 *     ),
	 *     "val2" => array(
	 *         "case" => array(
	 *             "Test" => "Fuji2",
	 *             "test" => "fuji2",
	 *             ""     => "empty2"
	 *         ),
	 *         "default" => "other2"
	 *     ),
	 * );
	 * 
	 * $data = array(
	 *     "val1" => "test",
	 *     "val2" => ""
	 * );
	 * $rts = YggTool::convCaseArray( $data, $catbl );
	 * 
	 * (Result)
	 * $rts => array( "val1" => "fuji1", "val2" => "empty2" );
	 * </code>
	 *
	 * @param array $data Target array
	 * @param array $catbl Convert information array<br />
	 * (It has array in array that has the following elements)<br />
	 * case => Case list in which the key is case of $value and value is replaced there<br />
	 * default => Default data if $value doesn't matches in $ctbl['case'].
	 * NULL is used instead of it when it is undefined.
	 * @return array
	 */
	public static function convCaseArray( $data, $catbl )
	{
		if( count($catbl) <= 0 ){
			return array();
		}

		$rts = array();
		foreach( $catbl as $key => $car ){
			if( !isset( $data[$key] ) ){
				$data[$key] = null;
			}

			$rts[$key] = self::convCase( $data[$key], $car );
		}

		return $rts;
	}


	/**
	 * Create page status
	 *
	 * This method calculates page status from $reccnt, $pgno, $recperpg.<br />
	 * This method throws RangeException when error occurs.
	 *
	 * <code>
	 * $rts = YggTool::statusPage( 1000, 3, 50, 7 );
	 * (Result)
	 * $rts => array(
	 *     "RC"      => 1000,
	 *     "PG"      => 3,
	 *     "PGMIN"   => 1,
	 *     "PGMAX"   => 200,
	 *     "PGRC"    => 50,
	 *     "PGRCS"   => 101,
	 *     "PGRCE"   => 150
	 * );
	 * </code>
	 *
	 * @param integer $reccnt Record count
	 * @param integer $pgno Current page no
	 * @param integer $recperpg Record count per page
	 * @return array Page status<br />
	 * RC      : All record count<br />
	 * PG      : Current page no<br />
	 * PGMIN   : Min page no<br />
	 * PGMAX   : Max page no<br />
	 * PGRC    : Record count in current page<br />
	 * PGRCS   : Starting record count of page<br />
	 * PGRCE   : Ending record count of page
	 */
	public static function statusPage( $reccnt, $pgno, $recperpg )
	{
		if( $reccnt < 0 ){
			throw new RangeException;
		}
		elseif( $recperpg < 1 ){
			throw new RangeException;
		}

		if( $reccnt == 0 ){
			return array(
				"rc"    => $reccnt,
				"pg"    => 0,
				"pgmin" => 0,
				"pgmax" => 0,
				"pgrc"  => 0,
				"pgrcs" => 0,
				"pgrce" => 0
			);
		}

		$pgmin = 1;

		$pgmax = (int)( $reccnt / $recperpg );
		if( $reccnt % $recperpg != 0 ){
			$pgmax++;
		}

		if( $pgno < 1 || $pgno > $pgmax ){
			return array(
				"rc"    => $reccnt,
				"pg"    => 0,
				"pgmin" => $pgmin,
				"pgmax" => $pgmax,
				"pgrc"  => 0,
				"pgrcs" => 0,
				"pgrce" => 0
			);
		}

		$pgrcs = ( $pgno - 1 ) * $recperpg + 1;

		$pgrce = $reccnt;
		if( $pgno != $pgmax ){
			$pgrce = $pgrcs + $recperpg - 1;
		}

		$pgrc = $pgrce - $pgrcs + 1;

		return array(
			"rc"      => $reccnt, 
			"pg"      => $pgno,
			"pgmin"   => $pgmin,
			"pgmax"   => $pgmax,
			"pgrc"    => $pgrc,
			"pgrcs"   => $pgrcs,
			"pgrce"   => $pgrce
		);
	}


	/**
	 * Get now time ( UNIX time stamp microtime format )
	 *
	 * This method returns now time formatted to unix time stamp and converts to float.<br />
	 * This method throws RuntimeException when error occurs.
	 *
	 * @return float
	 */
	public static function getUnixMicroTime()
	{
		$rts = explode( " ", microtime() ); 
		if( !is_array($rts) || count($rts) < 2 ){
			throw new RuntimeException;
		}

		return (float)$rts[1] + (float)$rts[0];
	}


	/**
	 * Create unique ID 
	 *
	 * This method creates unique ID.<br />
	 * Random seed is initialized.<br />
	 * RuntimeException is thrown when error occurs.
	 *
	 * @return string
	 */
	public static function uniqueID()
	{
		$rts = md5( uniqid( mt_rand(), true ) ); 
		if( !is_string($rts) ){
			throw new RuntimeException;
		}

		return $rts;
	}


	/**
	 * Create data byte size string
	 *
	 * This method converts from $size to string format of data byte size.<br />
	 * This method supports until MB. $data is less than 1000, returns 1KB string.<br />
	 * RuntimeException is thrown when error occurs.
	 *
	 * @param integer $size Target data
	 * @return string
	 */
	public static function convStrSize( $size )
	{
		if( $size < 0 ){
			throw new RangeException;
		}

		if( $size < 1024 ){
			return "1KB";
		}
		elseif( $size < 1048576 ){
			return number_format( (int)($size / 1024) ) . "KB";
		}

		return number_format( (int)($size / 1048576) ) . "MB";
	}


	/**
	 * Create string elements that range of integer numeric in array.
	 * 
	 * This method creates string elements that is numeric between $low to $high and
	 * adds header of $head and footer of $foot to them.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $rts = YggTool::rangeStrArray( 1, 5, "test" );
	 * (Result)
	 * $rts => (
	 *     "test1",
	 *     "test2",
	 *     "test3",
	 *     "test4",
	 *     "test5"
	 * );
	 * </code>
	 *
	 * @param integer $low Start point
	 * @param integer $high End point
	 * @param string $head Header
	 * @param string $foot Footer
	 *
	 * @return array
	 */
	public static function rangeStrArray(
		$low, $high, $head = "", $foot = ""
	)
	{
		if( $low > $high ){
			throw new UnexpectedValueException;
		}

		$rts = array();
		for( $i = $low; $i <= $high; $i++ ){
			$rts[] = $head . $i . $foot;
		}

		return $rts;
	}


	/**
	 * Create string elements that range of integer numeric in array.
	 * 
	 * This method creates string elements that is numeric between $low to $high.
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * <code>
	 * $rts = YggTool::rangeIntArray( 1, 5 );
	 * (Result)
	 * $rts => ( 1, 2, 3, 4, 5 )
	 * </code>
	 *
	 * @param integer $low Start point
	 * @param integer $high End point
	 * @return array
	 */
	public static function rangeIntArray( $low, $high )
	{
		if( $low > $high ){
			throw new UnexpectedValueException;
		}

		$rts = array();
		for( $i = $low; $i <= $high; $i++ ){
			$rts[] = $i;
		}

		return $rts;
	}
	

	/**
	 * Check IP address is included in network
	 * 
	 * This method checks $ip is included in
	 * $cidrl is specified array list of CIDR format data
	 * (xxx.xxx.xxx.xxx/subnetmask or/and ip address).<br />
	 * $cidrl is array and you can check some networks.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param string $ip IP address
	 * @param array $cidrl CIDR list
	 * @return boolean Returns TRUE if $ip is included in $cidr, otherwise returns FALSE.
	 */
	public static function matchNetAddr( $ip, $cidrl )
	{
		$ip = ip2long( $ip );
		if( !is_int($ip) ){
			throw new UnexpectedValueException;
		}
		elseif( count($cidrl) <= 0 ){
			throw new UnexpectedValueException;
		}

		foreach( $cidrl as $key => $cidr ){
			$cidr = explode( "/", $cidr );
			if( count($cidr) < 1 || count($cidr) > 2 ){
				throw new UnexpectedValueException;
			}

			$cidr[0] = ip2long( $cidr[0] );
			if( !is_int($cidr[0]) ){
				throw new UnexpectedValueException;
			}

			try{
				if( count($cidr) != 2 ){
					$cidr[1] = "32";
				}
				$cidr[1] = YggSafe::safeInt( $cidr[1] );
				if( $cidr[1] < 1 || $cidr[1] > 32 ){
					throw new UnexpectedValueException;
				}
			}
			catch( Exception $e ){
				throw new UnexpectedValueException;
			}

			$cidrl[$key] = $cidr;
		}

		foreach( $cidrl as $cidr ){
			$ipnet = $ip & ~( (1 << (32 - $cidr[1]) ) - 1 );
			$net   = $cidr[0] & ~( (1 << (32 - $cidr[1]) ) - 1 );
			if( $ipnet == $net ){
				return true;
			}
		}
			
		return false;
	}

	
	/**
	 * Search and call static method in the instance
	 * 
	 * This method searches static method specified $smn in $obj class sequentially
	 * from extended to parent class.
	 * The first matching static method is called with argument data specified by $arg.
	 * Then, this method is used call_user_func_array function.
	 *
	 * @param object $obj Target object
	 * @param string $smn Static method name (Must be public)
	 * @param array $arg Argument list
	 * @return mixed Return value of static method
	 */
	public static function callVirtualStatic( $obj, $smn, $arg = null )
	{
		if( $arg === null ){
			$arg = array();
		}

		$cnm = get_class( $obj );
		if( $cnm === false ){
			throw new UnexpectedValueException;
		}

		$cb = array( $cnm, $smn );
		if( is_callable( $cb ) ){
			return call_user_func_array( $cb, $arg );
		}

		while( true ){
			$cnm = get_parent_class( $cnm );
			if( !is_string( $cnm ) ){
				throw new UnexpectedValueException;
			}

			$cb = array( $cnm, $smn );
			if( is_callable( $cb ) ){
				break;
			}
		}

		return call_user_func_array( $cb, $arg );
	}

	
	/**
	 * Compare php version with current php version
	 * 
	 * This method compares current php version and $chkver.<br />
	 * This method throws UnexpectedValueException when error occurs.
	 *
	 * @param string $chkver Target version string
	 * @return integer Compare result<br />
	 * PAST : current version is past version than $chkver<br />
	 * SAME : current version and $chkver are same version<br />
	 * FUTURE : current version is future version than $chkver
	 */
	public static function cmpPHPVer( $chkver )
	{
		$rts = version_compare( phpversion(), $chkver );
		if( !is_int($rts) ){
			throw new UnexpectedValueException;
		}

		return $rts;
	}
}
?>
