<?php
/**
 * Directory whose name is unsigned integer creation class
 *
 * This file defines YggUIntNoDir class.
 * It includes methods to create directory whose name is unsigned integer.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggDirectory class
 */
require_once( "YggDore/Base/YggDirectory.php" );


/**
 * Directory whose name is unsigned integer creation class
 *
 * This class has methods to create directory whose name is unsigned integer.
 * Trouble that is directory creation limit over error can be escaped when
 * a lot of directory is created.
 *
 * <code>
 * $dirobj = new YggUIntNoDir;
 * $dirobj->set( 15 );
 * $dirno = $dirobj->get();
 * $dirpath = $dirobj->getPath();
 * $dirobj->create( "/home/yggcommon", 0755 );
 * </code>
 *
 * @package YggDore.Base
 */
class YggUIntNoDir {
	/**
	 * Directory No
	 */
	private $_dno;
	/**
	 * Directory path name
	 */
	private $_dpath;


	/**
	 * Constructor
	 *
	 * The instance is initialized.<br />
	 * Each arguments are specified initial value.
	 *
	 * @param integer $dno Directory name No (more than 0)
	 * @param string $id Directory name header (word before $dno)
	 */
	public function __construct( $dno = -1, $id = "" )
	{
		$this->_dno   = null;
		$this->_dpath = null;

		if( $dno == -1 && $id == "" ){
			return;
		}
		$this->set( $dno, $id );
	}


	/**
	 * Set directory name
	 *
	 * Directory name is set to the instance.
	 * $dno can be specified integer value more than 0.
	 * If $id is specified, directory format of first level is [id][dno].
	 * $id can be specified length whose length is less than 8.
	 *
	 * @param integer $dno Directory name No (more than 0)
	 * @param string $id Directory name header (word before $dno)
	 */
	public function set( $dno, $id = "" )
	{
		if( $dno < 0 ){
			throw new RangeException;
		}
		elseif( $id != "" ){
			if( strlen($id) > 8 || 
				!preg_match( '/^[a-zA-Z][a-zA-Z\-_]*$/', $id ) ){
				throw new UnexpectedValueException;
			}
		}

		$this->_dno      = $dno;
		$this->_dpath[0] = $id . (int)($dno / 1000000);
		$this->_dpath[1] = ((int)($dno / 1000)) % 1000;
		$this->_dpath[2] = $dno % 1000;
	}


	/**
	 * Get directory name
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return integer
	 */
	public function get()
	{
		if( $this->_dno === null ){
			throw new BadMethodCallException;
		}

		return $this->_dno;
	}


	/**
	 * Get directory path name
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return string
	 */
	public function getPath()
	{
		if( $this->_dpath === null ){
			throw new BadMethodCallException;
		}

		return $this->_dpath[0] . "/" .
		       $this->_dpath[1] . "/" .
			   $this->_dpath[2];
	}


	/**
	 * Create directory
	 *
	 * The directory is created in $tgtdir.<br />
	 * When $tgtdir isn't specified, it is created to current directory.<br />
	 * Umask influences this method.
	 *
	 * @param string $tgtdir Target directory
	 * @param integer $mode Permission
	 */
	public function createDir( $tgtdir = "", $mode = 0777 )
	{
		if( $this->_dpath === null ){
			throw new BadMethodCallException;
		}

		if( $mode < 0 || $mode > 0777 ){
			throw new RangeException;
		}

		if( $tgtdir != "" ){
			$tgtdir = YggDirectory::endSlash( $tgtdir );
		}

		$dirpath = $tgtdir;
		for( $i = 0; $i < count($this->_dpath); $i++ ){
			if( $i != 0 ){
				$dirpath .= "/";
			}
			$dirpath .= $this->_dpath[$i];

			if( file_exists( $dirpath ){
				continue;
			}
			$rts = mkdir( $dirpath, $mode );
			if( !$rts ){
				throw new RuntimeException;
			}
		}
	}
}
?>
