<?php
/**
 * Web control class
 *
 * This file defines YggWebControl class.
 * It includes methods to optimize data of web request data and so on.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );
/**
 * Require YggInt class
 */
require_once( "YggDore/Base/YggInt.php" );
/**
 * Require YggArray class
 */
require_once( "YggDore/Base/YggArray.php" );


/**
 * Web control class
 *
 * This class has methods to optimize data of web request data and so on.
 *
 * <code>
 * $hoge = "hoge ";
 * YggWebControl::text( $hoge );
 * </code>
 *
 * @package YggDore.Base
 */
class YggWebControl {
	/**
	 * Optimize text data
	 *
	 * This method optimizes (trim and so on) $data.
	 *
	 * @param string $data Target value
	 * @return string
	 */
	public static function text( $data )
	{
		return YggString::trim( YggString::eol( $data ) );
	}


	/**
	 * Optimize text data in array
	 *
	 * This method optimizes (trim and so on) $data specified target keys in $dktbl.
	 *
	 * @param array $data Target array
	 * @param array $dktbl Target key list
	 * @return array
	 */
	public static function textArray( $data, $dktbl )
	{
		$rts = YggArray::extract( $data, $dktbl );

		foreach( array_keys($rts) as $key ){
			$rts[$key] = self::text( $rts[$key] );
		}

		return $rts;
	}


	/**
	 * Optimize text data to type of integer
	 *
	 * This method optimizes (trim and so on) and converts (to integer) $data.<br />
	 * If $data is empty string, NULL is returned.
	 *
	 * @param string $data Target value
	 * @return integer
	 */
	public static function textInt( $data )
	{
		$data = self::text( $data );
		if( $data == "" ){
			return null;
		}
		return YggInt::conv( $data );
	}


	/**
	 * Optimize text data to type of integer in array
	 *
	 * This method optimizes (trim and so on) and
	 * converts $data specified target keys in $dktbl.<br />
	 * If each value in $data is empty string, NULL is returned.
	 *
	 * @param array $data Target array
	 * @param array $dktbl Target key list
	 * @return array
	 */
	public static function textIntArray( $data, $dktbl )
	{
		$rts = YggArray::extract( $data, $dktbl );

		foreach( array_keys($rts) as $key ){
			$rts[$key] = self::textInt( $rts[$key] );
		}

		return $rts;
	}


	/**
	 * Optimize text area data
	 *
	 * This method optimizes (trim and so on) $data.<br />
	 * $maxret is specified value that is allows number of repeat eol.
	 *
	 * @param string $data Target value
	 * @param integer $maxret Number of repeatable eol
	 * @return string
	 */
	public static function textArea( $data, $maxret = -1 )
	{
		return YggString::trim( YggString::eol( $data, $maxret ) );
	}


	/**
	 * Optimize text area data in array
	 *
	 * This method optimizes (trim and so on) $data specified target keys in $dktbl.<br />
	 * $maxret is specified value that is allows number of repeat eol.
	 *
	 * @param array $data Target value
	 * @param array $dktbl Target key list
	 * @param integer $maxret Number of repeatable eol
	 * @return array
	 */
	public static function textAreaArray( $data, $dktbl, $maxret = -1 )
	{
		$rts = YggArray::extract( $data, $dktbl );

		foreach( array_keys($rts) as $key ){
			$rts[$key] = self::textArea( $rts[$key], $maxret );
		}

		return $rts;
	}


	/**
	 * Optimize text data to type of boolean
	 *
	 * This method optimizes converts (to boolean) $data.<br />
	 * If $data is empty string, FALSE is returned, otherwize TRUE is returned.
	 *
	 * @param string $data Target value
	 * @return boolean
	 */
	public static function textBool( $data )
	{
		if( $data == "" ){
			return false;
		}

		return true;
	}
}
?>
