<?php
/**
 * Web request class
 *
 * This file defines YggWebRequest class.
 * It includes methods to get request data from client.
 *
 * @package YggDore.Base
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );
/**
 * Require YggRawWebRequest class
 */
require_once( "YggDore/Base/YggRawWebRequest.php" );


/**
 * Web request class
 *
 * This class has methods to get request data from client.
 *
 * <code>
 * $req = YggWebRequest::getRequest( array( "test" ), 1024 );
 * </code>
 *
 * @package YggDore.Base
 */
class YggWebRequest {
	/**
     * Get $_REQUEST
	 *
	 * This method extracts data matched with keys of $keyl in $_REQUEST and
	 * converts them to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $limit Limit string length
     * @return array
     */
	public static function getRequestArray( $keyl, $limit = YGG_YSSLIMIT )
	{
		return YggString::stableArray(
			YggRawWebRequest::getRequest(), $keyl, $limit
		);
	}


	/**
     * Get a data of $_REQUEST
	 *
	 * This method extracts data matched with $key in $_REQUEST and converts it to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $limit Limit string length
     * @return string
     */
	public static function getRequest( $key, $limit = YGG_YSSLIMIT )
	{
		$data = self::getRequestArray( array( $key ), $limit );
		return $data[$key];
	}


	/**
     * Get $_POST
	 *
	 * This method extracts data matched with keys of $keyl in $_POST and
	 * converts them to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $limit Limit string length
     * @return array
     */
	public static function getPostArray( $keyl, $limit = YGG_YSSLIMIT )
	{
		return YggString::stableArray(
			YggRawWebRequest::getPost(), $keyl, $limit
		);
	}


	/**
     * Get a data of $_POST
	 *
	 * This method extracts data matched with $key in $_POST and converts it to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $limit Limit string length
     * @return string
     */
	public static function getPost( $key, $limit = YGG_YSSLIMIT )
	{
		$data = self::getPostArray( array( $key ), $limit );
		return $data[$key];
	}


	/**
     * Get $_GET
	 *
	 * This method extracts data matched with keys of $keyl in $_GET and
	 * converts them to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $limit Limit string length
     * @return array
     */
	public static function getGetArray( $keyl, $limit = YGG_YSSLIMIT )
	{
		return YggString::stableArray(
			YggRawWebRequest::getGet(), $keyl, $limit
		);
	}


	/**
     * Get a data of $_GET
	 *
	 * This method extracts data matched with $key in $_GET and converts it to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $limit Limit string length
     * @return string
     */
	public static function getGet( $key, $limit = YGG_YSSLIMIT )
	{
		$data = self::getGetArray( array( $key ), $limit );
		return $data[$key];
	}


	/**
     * Get $_COOKIE
	 *
	 * This method extracts data matched with keys of $keyl in $_COOKIE and
	 * converts them to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $limit Limit string length
     * @return array
     */
	public static function getCookieArray( $keyl, $limit = YGG_YSSLIMIT )
	{
		return YggString::stableArray(
			YggRawWebRequest::getCookie(), $keyl, $limit
		);
	}


	/**
     * Get a data of $_COOKIE
	 *
	 * This method extracts data matched with $key in $_COOKIE and converts it to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $limit Limit string length
     * @return string
     */
	public static function getCookie( $key, $limit = YGG_YSSLIMIT )
	{
		$data = self::getCookieArray( array( $key ), $limit );
		return $data[$key];
	}


	/**
     * Get $_ENV
	 *
	 * This method extracts data matched with keys of $keyl in $_ENV and
	 * converts them to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $limit Limit string length
     * @return array
     */
	public static function getEnvArray( $keyl, $limit = YGG_YSSLIMIT )
	{
		return YggString::stableArray(
			YggRawWebRequest::getEnv(), $keyl, $limit
		);
	}


	/**
     * Get a data of $_ENV
	 *
	 * This method extracts data matched with $key in $_ENV and converts it to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $limit Limit string length
     * @return string
     */
	public static function getEnv( $key, $limit = YGG_YSSLIMIT )
	{
		$data = self::getEnvArray( array( $key ), $limit );
		return $data[$key];
	}


	/**
     * Get $_SERVER
	 *
	 * This method extracts data matched with keys of $keyl in $_SERVER and
	 * converts them to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $limit Limit string length
     * @return array
     */
	public static function getServerArray( $keyl, $limit = YGG_YSSLIMIT )
	{
		return YggString::stableArray(
			YggRawWebRequest::getServer(), $keyl, $limit
		);
	}


	/**
     * Get a data of $_SERVER
	 *
	 * This method extracts data matched with $key in $_SERVER and converts it to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $limit Limit string length
     * @return string
     */
	public static function getServer( $key, $limit = YGG_YSSLIMIT )
	{
		$data = self::getServerArray( array( $key ), $limit );
		return $data[$key];
	}


	/**
     * Get $_FILES
	 *
	 * This method extracts data matched with keys of $keyl in $_FILES and
	 * filename (name and tmp_name elements) is converted to safety.
     *
	 * @param array $keyl Target key list
	 * @param integer $nmlimit Limit string length of file name
     * @return array
     */
	public static function getFilesArray( $keyl, $nmlimit = YGG_YSSLIMIT )
	{
		if( $nmlimit < 0 ){
			throw new UnexpectedValueException;
		}

		$data = YggSafe::safeArray( YggRawWebRequest::getFiles() );
		$data = YggArray::extract( $data, $keyl );

		foreach( $keyl as $key ){
			if( $data[$key] === null ){
				continue;
			}
			elseif( $data[$key]['tmp_name'] == "" ||
			        $data[$key]['name'] == "" ){
				$data[$key] = null;
				continue;
			}

			$data[$key]['name'] = YggString::stable(
				$data[$key]['name'], $nmlimit
			);
		}

		return $data;
	}


	/**
     * Get a data of $_FILES
	 *
	 * This method extracts data matched with $key in $_FILES and
	 * filename (name and tmp_name elements) is converted to safety.
     *
	 * @param arraykey $key Target key name
	 * @param integer $nmlimit Limit string length of file name
     * @return array
     */
	public static function getFiles( $key, $nmlimit = YGG_YSSLIMIT )
	{
		$data = self::getFilesArray( array( $key ), $nmlimit );
		return $data[$key];
	}
}
?>
