<?php
/**
 * Common data checker
 *
 * This file defines YSDataCheck class.
 * It includes methods to check data.
 *
 * @package YggDore.Server.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggInt class
 */
require_once( "YggDore/Base/YggInt.php" );
/**
 * Require YggMessage class
 */
require_once( "YggDore/Base/YggMessage.php" );
/**
 * Require YggMailFormatter class
 */
require_once( "YggDore/Base/YggMailFormatter.php" );
/**
 * Require the server configuration file
 */
require_once( "YggDore/Server/Config/Config.php" );



/**
 * Common data checking class
 *
 * This class has methods to check data.
 *
 * @package YggDore.Server.Base
 */
class YSDataCheck {
	/**
	 * Check string
	 *
	 * This method checks $str format.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param string $str Target string
	 * @param integer $len Max length
	 * @param boolean $aes Flag to occur error when $val is empty
	 * @param arraykey $dek Key name to set message if error occurs
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $str is empty)<br />
	 * $dek => YGG_STS_EMPTY<br />
	 * (Case. $str is illegal data)<br />
	 * $dek => YGG_STS_FORMAT
	 */
	public static function str( $str, $len, $aes = true, $dek = "key", $msg = null )
	{
		$msg = YggMessage::mask( $msg, array( $dek ) );
		if( $msg !== null ){
			return $msg;
		}

		if( $len <= 0 ){
			throw new UnexpectedValueException;
		}
		elseif( $str == "" ){
			if( $aes ){
				return null;
			}
			return array( $dek => YGG_STS_EMPTY );
		}
		elseif( iconv_strlen($str) > $len ){
			return array( $dek => YGG_STS_FORMAT );
		}

		return null;
	}


	/**
	 * Check key value
	 *
	 * This method checks whether $key is greater or equal to YS_KEY_MIN.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param integer $key Target value
	 * @param arraykey $dek Key name to set message if error occurs
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $key is illegal data)<br />
	 * $dek => YGG_STS_FORMAT
	 */
	public static function key( $key, $dek = "key", $msg = null )
	{
		return self::numInt( $key, YS_KEY_MIN, YGG_INTMAX, $dek, $msg );
	}


	/**
	 * Check integer value
	 *
	 * This method checks whether $key is range of $min and $max.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param integer $val Target value
	 * @param integer $min Allow min value
	 * @param integer $max Allow max value
	 * @param arraykey $dek Key name to set message if error occurs
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $key is illegal data)<br />
	 * $dek => YGG_STS_FORMAT
	 */
	public static function numInt(
		$val, $min = YGG_INTMIN, $max = YGG_INTMAX, $dek = "val", $msg = null
	)
	{
		$msg = YggMessage::mask( $msg, array( $dek ) );
		if( $msg !== null ){
			return $msg;
		}

		if( $min > $max ){
			throw new UnexpectedValueException;
		}
		elseif( $val < $min ){
			return array( $dek => YGG_STS_FORMAT );
		}
		elseif( $val > $max ){
			return array( $dek => YGG_STS_FORMAT );
		}

		return null;
	}


	/**
	 * Check integer string
	 *
	 * This method checks whether $key is numeric and range of $min and $max.<br />
	 * When type of $key is string, the value is converted to integer and
	 * returned to $ival.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param integer &$ival Value stored converting result
	 * @param string $val Target value
	 * @param integer $min Allow min value
	 * @param integer $max Allow max value
	 * @param boolean $aes Flag to occur error when $val is empty
	 * @param arraykey $dek Key name to set message if error occurs
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $key is illegal data)<br />
	 * $dek => YGG_STS_FORMAT
	 */
	public static function strInt(
		&$ival,
		$val,
		$min = YGG_INTMIN,
		$max = YGG_INTMAX,
		$aes = true,
		$dek = "val",
		$msg = null
	)
	{
		$ival = null;

		$msg = YggMessage::mask( $msg, array( $dek ) );
		if( $msg !== null ){
			return $msg;
		}

		if( $val == "" ){
			if( $aes ){
				return array( $dek => YGG_STS_EMPTY );
			}
		}

		try{
			$val = YggInt::conv( $val );
		}
		catch( Exception $e ){
			return array( $dek => YGG_STS_FORMAT );
		}

		$rts = self::numInt( $val, $min, $max, $dek, $msg );
		if( $rts !== null ){
			return array( $dek => YGG_STS_FORMAT );
		}

		$ival = $val;

		return null;
	}


	/**
	 * Check password
	 *
	 * This method checks $ps format.<br />
	 * $ps isn't checked whether database data is as same as it.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param string $ps Password
	 * @param arraykey $dek Key name to set message if error occurs
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $ps is empty)<br />
	 * $dek => YGG_STS_EMPTY<br />
	 * (Case. $ps is illegal data)<br />
	 * $dek => YGG_STS_FORMAT
	 */
	public static function pass( $ps, $dek = "ps", $msg = null )
	{
		$msg = YggMessage::mask( $msg, array( $dek ) );
		if( $msg !== null ){
			return $msg;
		}

		$len = iconv_strlen( $ps );
		if( $len <= 0 ){
			return array( $dek => YGG_STS_EMPTY );
		}
		elseif( $len < 8 || $len > YS_STRING_STRMAX ){
			return array( $dek => YGG_STS_FORMAT );
		}
		elseif( !preg_match( '/^[[:graph:]]+$/', $ps ) ){
			return array( $dek => YGG_STS_FORMAT );
		}

		return null;
	}


	/**
	 * Check password and password to confirm
	 *
	 * This method checks $ps1 and $ps2.<br />
	 * They are checked format, $ps1 is as same as $ps2.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param string $ps1 Password1
	 * @param string $ps2 Password2
	 * @param arraykey $dekps1 Key name to set message
	 * if error occurs in $ps1
	 * @param arraykey $dekps2 Key name to set message
	 * if error occurs in $ps2
	 * @param array $msg Message
	 *
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $ps1 and $ps2 isn't same)<br />
	 * dekps1, dekps2 => YGG_STS_NOTSAME<br />
	 * (This method calls the following methods and merges message)<br />
	 * YSDataCheck::pass with $ps1, $ps2 (dekps1, dekps2)
	 */
	public static function passSame(
		$ps1, $ps2, $dekps1 = "ps1", $dekps2 = "ps2", $msg = null
	)
	{
		$derr = array();

		foreach( array( "ps1", "ps2" ) as $key ){
			$dek = "dek" . $key;

			$mmsg = YggMessage::mask( $msg, array( $$dek ) );
			if( $mmsg !== null ){
				$derr[$$dek] = $mmsg[$$dek];
				continue;
			}

			$rts = self::pass( $$key, $$dek, $msg );
			if( $rts !== null ){
				$derr[$$dek] = $rts[$$dek];
				continue;
			}
		}
		if( count($derr) > 0 ){
			return $derr;
		}
		if( $ps1 != $ps2 ){
			return array(
				$dekps1 => YGG_STS_NOTSAME,
				$dekps2 => YGG_STS_NOTSAME
			);
		}

		return null;
	}


	/**
	 * Check password and flag of password change
	 *
	 * This method checks format of $ps is password format when
	 * $isps is specified TRUE.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param boolean $isps Flag to check password
	 * @param string $ps Password
	 * @param arraykey $dekisps Key name to set message if error occurs in $isps
	 * @param arraykey $dekps Key name to set message if error occurs in $ps
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (This method calls the following methods and merges message)<br />
	 * YSDataCheck::pass with $ps (dekps)
	 */
	public static function passChg(
		$isps, $ps, $dekisps = "isps", $dekps = "ps", $msg = null
	)
	{
		$mmsg = YggMessage::mask( $msg, array( $dekisps ) );
		if( $mmsg !== null ){
			return $mmsg;
		}

		if( !$isps ){
			return null;
		}

		return self::pass( $ps, $dekps, $msg );
	}


	/**
	 * Check password password to confirm, flag of change password
	 *
	 * This method checks format of $ps1 and $ps2 is password format when
	 * $isps is specified TRUE.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param boolean $isps Flag to check password
	 * @param string $ps1 Password
	 * @param string $ps2 Password again
	 * @param arraykey $dekisps Key name to set message if error occurs
	 * in $isps
	 * @param arraykey $dekps1 Key name to set message if error occurs
	 * in $ps1
	 * @param arraykey $dekps2 Key name to set message if error occurs
	 * in $ps2
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (This method calls the following methods and merges message)<br />
	 * YSDataCheck::passSame with $ps1 and $ps2 (dekps1, dekps2)
	 */
	public static function passChgSame(
		$isps, $ps1, $ps2,
		$dekisps = "isps", $dekps1 = "ps1", $dekps2 = "ps2",
		$msg = null
	)
	{
		$mmsg = YggMessage::mask( $msg, array( $dekisps ) );
		if( $mmsg !== null ){
			return $mmsg;
		}

		if( !$isps ){
			return null;
		}

		return self::passSame( $ps1, $ps2, $dekps1, $dekps2, $msg );
	}


	/**
	 * Check email address
	 *
	 * This method checks format of $mail is e-mail format.<br />
	 * Returns message type of array if error occurs,
	 * otherwise returns NULL.
	 *
	 * @param string $mail Email address
	 * @param arraykey $dek Key name to set message if error occurs
	 * @param array $msg Message
	 * @return array_or_null Message<br />
	 * Return NULL if error doesn't occur, otherwise the following data is returned.<br />
	 * (Case. $mail is empty)<br />
	 * $dek => YGG_STS_EMPTY<br />
	 * (Case. $mail is illegal data)<br />
	 * $dek => YGG_STS_FORMAT
	 */
	public static function mail( $mail, $dek = "mail", $msg = null )
	{
		$msg = YggMessage::mask( $msg, array( $dek ) );
		if( $msg !== null ){
			return $msg;
		}

		if( $mail == "" ){
			return array( $dek => YGG_STS_EMPTY );
		}
		$rts = YggMailFormatter::checkGlobalAddr( $mail );
		if( !$rts ){
			return array( $dek => YGG_STS_FORMAT );
		}

		return null;
	}
}
?>
