<?php
/**
 * Session management class
 *
 * This file defines YSSession class.
 * It includes methods to control session.
 *
 * @package YggDore.Server.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggSessionMySQLi class
 */
require_once( "YggDore/Base/YggSessionMySQLi.php" );
/**
 * Require the server configuration file
 */
require_once( "YggDore/Server/Config/Config.php" );
/**
 * Require YSMySQLi class
 */
require_once( "YggDore/Server/Base/YSMySQLi.php" );



/**
 * Session management class
 *
 * This class has methods to control session.
 *
 * <code>
 * $login = new YSSession;
 * $login->start();
 * $login->stop();
 * </code>
 *
 * @package YggDore.Server.Base
 */
class YSSession {
	/**
	 * MySQL connection
	 */
	private $_mysqli;
	/**
	 * Flag to close mysql connection
	 */
	private $_iscms;
	/**
	 * Session
	 */
	private $_sess;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __constructor()
	{
		$this->_mysqli = null;
		$this->_iscms  = null;
		$this->_sess   = null;
	}
	
	
	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_mysqli === null ){
			return false;
		}

		return true;
	}


	/**
	 * Get mysqli connection
	 *
	 * This method returns value of the instance.<br />
	 * BadMethodCallException is thrown when error occurs.
	 *
	 * @return YggMySQLi
	 */
	public function getMySQLi()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_mysqli;
	}


	/**
	 * Get session
	 *
	 * This method returns value of the instance.<br />
	 * BadMethodCallException is thrown when error occurs.
	 *
	 * @return YggSessionMySQLi
	 */
	public function getSession()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_sess;
	}


	/**
	 * Get session ID
	 *
	 * This method returns value of the instance.<br />
	 * BadMethodCallException is thrown when error occurs.
	 *
	 * @return YggSessionID
	 */
	public function getSessionID()
	{
		return $this->getSession()->getID();
	}


	/**
	 * Get session data area
	 *
	 * This method returns value of the instance.<br />
	 * BadMethodCallException is thrown when error occurs.
	 *
	 * @return mixed
	 */
	public function &getSDA()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_sess->getDataArea();
	}


	/**
	 * Get all instance data
	 *
	 * This method returns value of the instance.<br />
	 * BadMethodCallException is thrown when error occurs.
	 *
	 * @return array<br />
	 * (It has the following elements)<br />
	 * mysqli : (YggMySQLi) MySQLi connection<br />
	 * sess : (YggSession) Session data
	 */
	public function get()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return array(
			"mysqli" => $this->_mysqli,
			"sess"   => $this->_sess
		);
	}

	
	/**
	 * Start session
	 *
	 * This method connects to database and session is started.<br />
	 * You must call YSSession::stop method later.
	 * Session data is stored to database when it is called.
	 *
	 * @param YggMySQLi $mysqli Database connection<br />
	 * Use default connection when this argument is specified NULL.
	 * @param array $conf Session configuration<br />
	 * The array of argument has merging $conf of YggSession::start method and
	 * the following element.<br />
	 * tbn : (string) Table name<br />
	 * cln : (integer) Number of seconds that is time of session log keeping
	 */
	public function start( $mysqli = null, $conf = null )
	{
		if( $conf === null ){
			$conf = self::getDefaultConfig();
		}

		$iscms = false;
		if( $mysqli === null ){
			$iscms  = true;
			$mysqli = YSMySQLi::connect();
		}

		$sess = null;
		try{
			$svr = YggWebRequest::getServerArray(
				array( YS_RAKEY, "HTTP_USER_AGENT" ), 1024
			);

			$sess = new YggSessionMySQLi( $mysqli, $conf );
			$sess->cleanData( $conf['cln'] + $conf['dct'] );
			$sess->start();
		}
		catch( Exception $e ){
			if( $iscms ){
				$rts = $mysqli->close();
				if( !$rts ){
					throw new RuntimeException;
				}
			}
			throw $e;
		}

		$this->_mysqli = $mysqli;
		$this->_iscms  = $iscms;
		$this->_sess   = $sess;
	}

	
	/**
	 * Stop login
	 *
	 * This method must call after YSSessionBase::start method.
	 * Session data is saved, disconnected database connection and cleared member variable.
	 * This method returns session ID.
	 *
	 * @return YggSessionID
	 */
	public function stop()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$sid = $this->_sess->getID();

		$this->_sess->stop();

		if( $this->_iscms ){
			$rts = $this->_mysqli->close();
			if( !$rts ){
				throw new RuntimeException;
			}
		}

		$this->_mysqli = null;
		$this->_iscms  = null;
		$this->_sess   = null;

		return $sid;
	}

	
	/**
	 * Destroy login session
	 *
	 * This method must call after YSSessionBase::start method.
	 * Current session is deleted.
	 */
	public function delete()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$this->_sess->delete();

		if( $this->_iscms ){
			$rts = $this->_mysqli->close();
			if( !$rts ){
				throw new RuntimeException;
			}
		}

		$this->_mysqli = null;
		$this->_iscms  = null;
		$this->_sess   = null;
	}


	/**
	 * Get default configuration
	 *
	 * This method returns default configuration data.
	 *
	 * @return array Configuration data<br />
	 * (Same as $conf of YSSession::start method)
	 */
	public static function getDefaultConfig()
	{
		return array_merge(
			YggSessionMySQLi::getDefaultConfig(),
			array(
				"tbn" => "SESSION",
				"snm" => YS_SESSION_NAME,
				"cln" => YS_SESSION_CLSEC
			)
		);
	}
}
?>
