<?php
/**
 * Class to manage web request data by session
 *
 * This file defines YSSessionData class.
 * It includes methods to manage web request data by session.
 *
 * @package YggDore.Server.Base
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YSSession class
 */
require_once( "YggDore/Server/Base/YSSession.php" );



/**
 * Class to manage web request data by session
 *
 * This class has methods to manage web request data by session.
 *
 * <code>
 * $sess = new YSSession;
 * $sess->start();
 * $sdc  = new YSSessionData( "hogehoge" );
 * $req  = array( "test1" => 2, "test2" => 3 );
 * $sdc->set( $req );
 * $sdc->save( $sess );
 * $sess->stop();
 * </code>
 *
 * @package YggDore.Server.Base
 */
class YSSessionData {
	/**
	 * Default key name for save data to session
	 */
	const SKEY_DEF = "sd";


	/**
	 * Key name of session array to save data
	 */
	private $_skey;
	/**
	 * Data
	 */
	private $_data;


	/**
	 * Constructor
	 *
	 * The instance is initialized.<br />
	 * Each arguments are specified initial value.
	 *
	 * @param arraykey $skey Key name of session array to save data
	 */
	public function __construct( $skey = self::SKEY_DEF )
	{
		$this->clear();
		$this->_skey = $skey;
	}
	

	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_data = null;
	}


	/**
	 * Get session key name
	 *
	 * This method returns value of the instance.
	 *
	 * @return arraykey
	 */
	public function getSKey()
	{
		return $this->_skey;
	}


	/**
	 * Get data
	 *
	 * This method returns value of the instance.
	 *
	 * @return mixed
	 */
	public function get()
	{
		return $this->_data;
	}


	/**
	 * Extract data
	 *
	 * This method extracts and gets data in instance.<br />
	 * If the instance doesn't has some elements specified by $keyl,
	 * NULL is set to them.
	 *
	 * @param array $keyl Extract array key list
	 * @return array
	 */
	public function extract( $keyl )
	{
		if( !is_array( $this->_data ) && !is_null( $this->_data ) ){
			throw new BadMethodCallException;
		}

		$rts = array();
		foreach( $keyl as $key ){
			if( !isset( $this->_data[$key] ) ){
				$rts[$key] = "";
				continue;
			}

			$rts[$key] = $this->_data[$key];
		}

		return $rts;
	}


	/**
     * Set data
	 *
	 * $data is set to the instance.
	 *
	 * @param mixed $data
     */
	public function set( $data )
	{
		$this->_data = $data;
	}


	/**
	 * Load data from session
	 *
	 * Data is set from session to the instance.
	 *
	 * @param YSSession $sess Session
	 */
	public function load( $sess )
	{
		$sd =& $sess->getSDA();
		if( !isset( $sd[$this->_skey] ) ){
			$this->_data = null;
			return;
		}
		$this->_data = $sd[$this->_skey];
	}


	/**
	 * Save data to session
	 *
	 * Current data in the instance is transfered to session data area.
	 *
	 * @param YSSession $sess Session
	 */
	public function save( $sess )
	{
		$sd =& $sess->getSDA();
		if( $sd === null ){
			$sd = array();
		}
		$sd[$this->_skey] = $this->_data;
	}
}
?>
