<?php
/**
 * Class to access to a YggDore Sky Gate interface url
 *
 * This file defines YSGClient.
 * It includes methods to access to interface url and get ID.
 *
 * @package YggDore.SkyGate
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggWebRequest class
 */
require_once( "YggDore/Base/YggWebRequest.php" );
/**
 * Require YggHTTPClient class
 */
require_once( "YggDore/Base/YggHTTPClient.php" );
/**
 * Require YSGURL class
 */
require_once( "YggDore/SkyGate/YSGURL.php" );
/**
 * Require YSGSecurity class
 */
require_once( "YggDore/SkyGate/YSGSecurity.php" );



/**
 * Class to access to a YggDore Sky Gate interface url
 *
 * This class has methods to access to interface url and get ID.
 *
 * <code>
 * $id = YSGClient::cert(
 *     "http://www.yggdore.com/key/?" .
 *     "%40cip=%40YCIP%40&%40cua=%40YCUA%40&%40wn=%40YWN%40"
 * );
 * </code>
 *
 * @package YggDore.SkyGate
 */
class YSGClient {
	/**
	 * Access to interface url and get ID
	 *
	 * This method accesses to interface url of sky gate specified by $ysi and
	 * returns ID.
	 *
	 * @param string $ysi YggDore sky gate interface url
	 * @param array $data Interface data<br />
	 * (It has the following elements)<br />
	 * YCIP => Client IP address<br />
	 * YCUA => Client User agent<br />
	 * YWN  => World name
	 * @return string ID
	 */
	public static function cert( $ysi )
	{
		$svr = YggWebRequest::getServerArray(
			array( "REMOTE_ADDR", "HTTP_USER_AGENT" )
		);

		$res = self::certRaw(
			$ysi,
			array(
				"YCIP" => $svr['REMOTE_ADDR'],
				"YCUA" => $svr['HTTP_USER_AGENT'],
				"YWN"  => ""
			)
		);
		return $res['id'];
	}


	/**
	 * Access to interface url and get result information
	 *
	 * This method accesses to interface url of sky gate specified by $ysi and
	 * returns ID, interface url information and xml data recieved from sky gate server.
	 *
	 * @param string $ysi YggDore sky gate interface url
	 * @param array $data Interface data<br />
	 * (It has the following elements)<br />
	 * YCIP => Client IP address<br />
	 * YCUA => Client User agent<br />
	 * YWN  => World name
	 * @return array Result information<br />
	 * (It has the following elements)<br />
	 * id  => (string) ID<br />
	 * key => (string) Key part in ID<br />
	 * world => (string) World part in ID<br />
	 * res => (SimpleXML) XML parser<br />
	 * ysi => (YSGURL) Status of $ysi
	 */
	public static function certRaw( $ysi, $data = NULL )
	{
		$ysiu = new YSGURL;
		$ysiu->set( $ysi );

		$xml = self::_reqHTTPXML(
			$ysiu->getActionURL(),
			$ysiu->embedQuery( $data )
		);
		if( $xml->getName() != "ID" ){
			throw new UnexpectedValueException;
		}

		$xmld = (array)$xml;

		if( !isset($xmld['Key']) ){
			throw new UnexpectedValueException;
		}
		elseif( strpos( $xmld['Key'], "@" ) !== false ){
			throw new UnexpectedValueException;
		}
		$key = (string)$xmld['Key'];
		if( $key == "" ){
			throw new UnexpectedValueException;
		}

		$urli  = $ysiu->getURLInfo();
		$host  = $urli['host'];
		$world = "";
		if( isset($xmld['World']) ){
			$world = $xmld['World'];
		}

		if( $world != "" ){
			// Note:
			// World name must match
			// right (high domain level) the order.
			$rts = YSGSecurity::checkWorld( $world, $host );
			if( !$rts ){
				throw new UnexpectedValueException;
			}
		}
		else{
			$world = $host;
		}

		return array(
			"id"    => $key . "@" . $world,
			"key"   => $key,
			"world" => $world,
			"res"   => $xml,
			"ysi"   => $ysiu
		);
	}


	/**
	 * Send POST request
	 *
	 * @param string $url Destination URL
	 * @param array $data Query data
	 * @return SimpleXMLElement XML object
	 */
	private static function _reqHTTPXML( $url, $data = null )
	{
		$hc   = YggHTTPClient::open( $url, $data, true );
		$data = $hc->get( 8192 );

		$xmle = @simplexml_load_string( $data );
		if( !is_a( $xmle, "SimpleXMLElement" ) ){
			throw new UnexpectedValueException;
		}

		return $xmle;
	}
}
?>
