<?php
/**
 * Class to check data about YggDore Sky Gate
 *
 * This file defines YSGSecurity.
 * It includes methods to check data about YggDore Sky Gate.
 *
 * @package YggDore.SkyGate
 * @author YggDore Co.,Ltd.
 */



/**
 * Require YggTool class
 */
require_once( "YggDore/Base/YggTool.php" );



/**
 * Class to check data about YggDore Sky Gate
 *
 * This class has methods to check data about YggDore Sky Gate.
 *
 * <code>
 * YSGSecurity::checkWorld(
 *     "hogehoge.com",
 *     "www.hogehoge.com"
 * );
 * </code>
 *
 * @package YggDore.SkyGate
 */
class YSGSecurity {
	/**
	 * Access control mode : None
	 */
	const ACM_NONE = 1;
	/**
	 * Access control mode : Auto mode
	 */
	const ACM_AUTO = 2;
	/**
	 * Access control mode : Manual mode
	 */
	const ACM_MANUAL = 3;


	/**
	 * Check world name
	 *
	 * This method checks world specified by $world is domain and
	 * it includes in host specified by $host.
	 *
	 * @param string $world World
	 * @param string $host Host name
	 * @return boolean
	 */
	public static function checkWorld( $world, $host )
	{
		if( $world == "" || $host == "" ){
			return false;
		}

		$wlen = strlen( $world );
		$w = substr( $host, $wlen * -1 );
		if( $w != $world ){
			return false;
		}
		elseif( $wlen != strlen($host) ){
			$w = substr( $host, ($wlen + 1) * -1, 1 );
			if( $w != "." ){
				return false;
			}
		}

		return true;
	}


	/**
	 * Check auto access control setting
	 *
	 * This method checks whether $ipc and $ips are the same.
	 * $ipc is current IP address.
	 * Normally, $ips is IP address at the login time.<br />
	 *
	 * @param string $ipc Client IP address
	 * @param string $ips IP address of saving point
	 * @param boolean
	 */
	public static function checkACAuto( $ipc, $ips )
	{
		$ipc = ip2long( $ipc );
		if( $ipc === FALSE ){
			throw new UnexpectedValueException;
		}

		$ips = ip2long( $ips );
		if( $ips === FALSE ){
			throw new UnexpectedValueException;
		}

		if( $ipc != $ips ){
			return false;
		}

		return true;
	}


	/**
	 * Check manual access control setting
	 *
	 * This method checks whether $ip is in network specified by $net.<br />
	 * $netl is specified array list of CIDR format data
	 * (xxx.xxx.xxx.xxx/subnetmask or/and ip address).
	 *
	 * @param string $ip Client IP address
	 * @param array $netl Access control network list
	 * @param boolean
	 * @see YggTool::matchNetAddr
	 */
	public static function checkACManual( $ip, $netl )
	{
		return YggTool::matchNetAddr( $ip, $netl );
	}


	/**
	 * Check access control setting
	 *
	 * This method checks client IP matches with access control setting.
	 * When $mode is ACM_NONE, this method doesn't control access.<br />
	 * When $mode is ACM_AUTO, this method checks whether $ipc and $ips are the same.
	 * $ipc is current IP address.
	 * Normally, $ips is IP address at the login time.<br />
	 * When $mode is ACM_MANUAL, this method checks whether
	 * $cip is in network specified by $netl.
	 * $netl is specified array list of CIDR format data
	 * (xxx.xxx.xxx.xxx/subnetmask or/and ip address).
	 *
	 * @param integer $mode Access control mode<br />
	 * (It is one of the following value)<br />
	 * ACM_NONE : No access control mode<br />
	 * ACM_AUTO : Auto access control mode<br />
	 * ACM_MANUAL : Manual access control mode
	 * @param string $ipc Current IP address
	 * @param string $ips IP address of saving point
	 * @param string $netl Access control network list
	 * @param boolean
	 */
	public static function checkAC( $mode, $ipc, $ips, $netl )
	{
		if( $mode == self::ACM_NONE ){
			return true;
		}
		elseif( $mode == self::ACM_AUTO ){
			return self::checkACAuto( $ipc, $ips );
		}

		return self::checkACManual( $ipc, $netl );
	}
}
?>
