<?php
/**
 * YggDore URL class
 *
 * This file defines YSGURL class.
 * It includes methods to support analizing and checking extended url.
 *
 * @package YggDore.SkyGate
 * @author YggDore Co.,Ltd.
 */


/**
 * Require YggSafe class
 */
require_once( "YggDore/Base/YggSafe.php" );
/**
 * Require YggString class
 */
require_once( "YggDore/Base/YggString.php" );


/**
 * YggDore URL class
 *
 * This class includes methods to analize and check extended url.
 * When the first character in the key name of request is atmark(@),
 * enclosed by atmarks in value can replace to other.
 *
 * <code>
 * $gurl = new YSGURL;
 * $gurl->set( "http://www.yggdore.com/hogehoge/?@yk=here@YKEY@here" );
 * $egu = $gurl->embedQuery(
 *     array( "YKEY" => "hogehoge" )
 * );
 * - Result -
 * $egu = array(
 *     "yk" => "herehogehogehere"
 * )
 * </code>
 *
 * @package YggDore.SkyGate
 */
class YSGURL {
	/**
	 * Raw URL
	 */
	private $_url;
	/**
	 * URL information
	 */
	private $_urli;
	/**
	 * Request query
	 */
	private $_que;


	/**
	 * Constructor
	 *
	 * The instance is initialized.
	 */
	public function __construct()
	{
		$this->clear();
	}


	/**
	 * Clear
	 *
	 * Member variables of the instance is initialized.
	 */
	public function clear()
	{
		$this->_url  = null;
		$this->_urli = null;
		$this->_que  = null;
	}


	/**
	 * Check instance data
	 *
	 * This method checks whether member variables are valid.<br />
	 * This method returns TRUE when they are valid, otherwise returns FALSE.
	 *
	 * @return boolean
	 */
	public function isSetIns()
	{
		if( $this->_url === null ){
			return false;
		}

		return true;
	}


	/**
	 * Get URL
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return string
	 */
	public function getURL()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_url;
	}


	/**
	 * Get URL information
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array Same as return value of parse_url function.
	 */
	public function getURLInfo()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_urli;
	}


	/**
	 * Get exclueded query URL
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array Same as return value of parse_str function.
	 */
	public function getActionURL()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$atm = "";
		$url = $this->_urli['scheme'] . "://";
		if( isset($this->_urli['user']) ){
			$url .= $this->_urli['user'] . ":" . $this->_urli['pass'];
			$atm  = "@";
		}
		$url .= $atm . $this->_urli['host'] . $this->_urli['path'];
		if( isset($this->_urli['fragment']) ){
			$url .= "#" . $this->_urli['fragment'];
		}

		return $url;
	}


	/**
	 * Get URL query
	 *
	 * This method returns value of the instance.<br />
	 * This method throws BadMethodCallException when error occurs.
	 *
	 * @return array Query data<br />
	 * (Its each records has the following elements)<br />
	 * key : (string) Query key name<br />
	 * value : (string) Query value
	 */
	public function getQuery()
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		return $this->_que;
	}


	/**
	 * Embed query data
	 *
	 * When the first character in the key name of query in
	 * this instance is atmark(@),
	 * enclosed by atmarks in value can replace to other,
	 * otherwise the target data isn't replaced.
	 * The pair of target word and replacing word is specified by $data.
	 * If this method success processing,
	 * replaced query data array is returned,
	 * otherwise exception is thrown.
	 *
	 * @param array $data The pair set of target word and replacing word
	 * @return array Result<br />
	 * (Its each records has the following elements)<br />
	 * key : (string) Query key name<br />
	 * value : (string) Query value
	 */
	public function embedQuery( $data )
	{
		if( !$this->isSetIns() ){
			throw new BadMethodCallException;
		}

		$rts = array();
		foreach( $this->_que as $qr ){
			if( substr( $qr['key'], 0, 1 ) != "@" ){
				$qr['value'] = $qr['value'];

				$rts[] = $qr;

				continue;
			}

			$qr['key']   = substr( $qr['key'], 1 );
			$qr['value'] = self::_embed( $qr['value'], $data );

			$rts[] = $qr;
		}

		return $rts;
	}


	/**
	 * Embed query string ( URL encode format )
	 *
	 * When the first character in the key name of query in
	 * this instance is atmark(@),
	 * enclosed by atmarks in value can replace to other,
	 * otherwise the target data isn't replaced.
	 * The pair of target word and replacing word is specified by $data.
	 * If this method success processing,
	 * replaced query data converted to url encode and return it,
	 * otherwise exception is thrown.
	 *
	 * @param array $data The pair set of target word and replacing word
	 * @return string URL encoding string
	 */
	public function embedQueryURL( $data )
	{
		$data = $this->embedQuery( $data );
		
		$rts = "";
		$sp  = "";
		foreach( $data as $qr ){
			$rts .= $sp .
			        urlencode( $qr['key'] ) .
			        '=' .
					urlencode( $qr['value'] );
			$sp   = "&";
		}

		return $rts;
	}


	/**
	 * Embed query string ( URL encode format )
	 *
	 * When the first character in the key name of query in
	 * this instance is atmark(@),
	 * enclosed by atmarks in value can replace to other,
	 * otherwise the target data isn't replaced.
	 * The pair of target word and replacing word is specified by $data.
	 * If this method success processing,
	 * replaced query data converted to html hidden tags and return it,
	 * otherwise exception is thrown.
	 *
	 * @param array $data The pair set of target word and replacing word
	 * @param boolean $isx If true, create XHTML tag
	 * @return string HTML hidden tag string
	 */
	public function embedQueryHTag( $data, $isx = true )
	{
		$data = $this->embedQuery( $data );

		$xtag = "";
		if( $isx ){
			$xtag = " /";
		}

		$rts = "";
		foreach( $data as $qr ){
			$rts .= '<input type="hidden" name="' .
			        YggHTML::specialChars( $qr['key'] ) .
			        '" value="' .
			        YggHTML::specialChars( $qr['value'] ) .
					'"' . $xtag . '>';
		}

		return $rts;
	}


	/**
	 * Set URL
	 *
	 * @param string $url URL
	 */
	public function set( $url )
	{
		$data = self::_parse( $url );

		$this->_url  = $data['url'];
		$this->_urli = $data['urli'];
		$this->_que  = $data['que'];
	}


	/**
	 * Check and parse url
	 *
	 * Check whether argument data is safety and parse url.
	 *
	 * @param string $url URL
	 * @return array Parse information<br />
	 * (It has the following elements)<br />
	 * url : (string) Same as URL<br />
	 * urli : (array) Same as return value of parse_url<br />
	 * que : (array) Same as return value of parse_str
	 */
	private static function _parse( $url )
	{
		if( $url == "" ){
			throw new UnexpectedValueException;
		}
		
		$urli = @parse_url( $url );
		if( !is_array( $urli ) ){
			throw new UnexpectedValueException;
		}

		$cet = array( "scheme", "host" );
		foreach( $cet as $key ){
			if( !isset( $urli[$key] ) || $urli[$key] == "" ){
				throw new UnexpectedValueException;
			}
		}

		$uit   = array( "user", "pass" );
		$uinec = 0;
		foreach( $uit as $key ){
			if( isset( $urli[$key] ) && $urli[$key] != "" ){
				$uinec++;
			}
		}
		if( $uinec > 0 && $uinec < count($uit) ){
			throw new UnexpectedValueException;
		}

		$urli['scheme'] = strtolower($urli['scheme']);
		if( $urli['scheme'] != "http" && $urli['scheme'] != "https" ){
			throw new UnexpectedValueException;
		}

		if( !isset( $urli['path'] ) || $urli['path'] == "" ){
			$urli['path'] = "/";
		}

		if( !isset( $urli['port'] ) ){
			$urli['port'] = 80;
			if( $urli['scheme'] == "https" ){
				$urli['port'] = 443;
			}
		}
		else{
			$urli['port'] = YggSafe::safeInt( $urli['port'] );
		}

		if( !isset( $urli['query'] ) ){
			$urli['query'] = "";
		}
		$que = self::_parseQuery( $urli['query'] );

		return array(
			"url"  => $url,
			"urli" => $urli,
			"que"  => $que
		);
	}


	/**
	 * Split string
	 *
	 * This method splits $str by $search.<br />
	 * $sidx is starting position, the first character is 0.
	 *
	 * @param string $str Target string
	 * @param string $search Search and split word
	 * @param integer $sidx Starting position
	 * @return string Result
	 */
	private static function _split( $str, $search, $sidx )
	{
		$eidx = strpos( $str, $search, $sidx );
		if( $eidx === false ){
			return substr( $str, $sidx );
		}

		return substr( $str, $sidx, $eidx - $sidx );
	}


	/**
	 * Parse query
	 *
	 * This method parses query string and returns array data.
	 *
	 * @param string $query Query
	 * @return array Result<br />
	 * (Its each records has the following elements)<br />
	 * key : (string) Query key name<br />
	 * value : (string) Query value
	 */
	private static function _parseQuery( $query )
	{
		$rts  = array();
		$sidx = 0;
		$qlen = strlen( $query );
		while( $sidx < $qlen ){
			$rec = self::_split( $query, "&", $sidx );

			$sidx += strlen($rec) + 1;

			if( $rec == "" ){
				continue;
			}

			$key = self::_split( $rec, "=", 0 );
			if( $key == "" ){
				continue;
			}

			$val = substr( $rec, strlen($key) + 1 );

			$rts[] = array(
				"key"   => urldecode( $key ),
				"value" => urldecode( $val )
			);
		}

		return $rts;
	}


	/**
	 * Embed words
	 *
	 * Embed words to enclosed by atmark characters.
	 *
	 * @param string $src Target string
	 * @param array $data The pair set of target word and replacing word
	 * @return string Result
	 */
	private static function _embed( $src, $data )
	{
		$data[''] = "@";
		return (string)preg_replace(
			'/@([^@]*)@/e',
			'isset($data[\'$1\']) ? $data[\'$1\'] : ""',
			$src
		);
	}


	/**
	 * Check url
	 *
	 * Check whether argument data is safety.
	 *
	 * @param string $url URL to outside service
	 * @return boolean If data is safety, return TRUE,
	 * Otherwise return FALSE
	 */
	public static function check( $url )
	{
		try{
			self::_parse( $url );
		}
		catch( Exception $e ){
			return false;
		}

		return true;
	}
}
?>
